<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Backup;
use App\Helpers\BackupHelper;

class BackupListCommand extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'backup:list 
                            {--type= : Filter by backup type (database, files, full)}
                            {--status= : Filter by status (pending, processing, completed, failed)}
                            {--limit=20 : Limit number of results}';

    /**
     * The console command description.
     */
    protected $description = 'List all backups';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $query = Backup::with('creator')->orderBy('created_at', 'desc');

        // Apply filters
        if ($type = $this->option('type')) {
            if (!in_array($type, ['database', 'files', 'full'])) {
                $this->error('Invalid type. Must be: database, files, or full');
                return 1;
            }
            $query->where('type', $type);
        }

        if ($status = $this->option('status')) {
            if (!in_array($status, ['pending', 'processing', 'completed', 'failed', 'cancelled'])) {
                $this->error('Invalid status. Must be: pending, processing, completed, failed, or cancelled');
                return 1;
            }
            $query->where('status', $status);
        }

        $limit = (int) $this->option('limit');
        $backups = $query->limit($limit)->get();

        if ($backups->isEmpty()) {
            $this->info('No backups found.');
            return 0;
        }

        // Prepare table data
        $headers = ['ID', 'Name', 'Type', 'Status', 'Size', 'Created', 'Created By'];
        $rows = [];

        foreach ($backups as $backup) {
            $rows[] = [
                $backup->id,
                $backup->original_name,
                ucfirst($backup->type),
                ucfirst($backup->status),
                $backup->file_size ? BackupHelper::formatBytes($backup->file_size) : '-',
                $backup->created_at->format('Y-m-d H:i:s'),
                $backup->creator->account_holder ?? 'System'
            ];
        }

        $this->table($headers, $rows);

        // Show summary
        $this->info("\nShowing {$backups->count()} backup(s)");
        
        // Show statistics
        $stats = [
            'Total' => Backup::count(),
            'Completed' => Backup::where('status', 'completed')->count(),
            'Failed' => Backup::where('status', 'failed')->count(),
            'Processing' => Backup::where('status', 'processing')->count(),
        ];

        $this->info("\nStatistics:");
        foreach ($stats as $label => $count) {
            $this->line("  {$label}: {$count}");
        }

        return 0;
    }
}