<?php

namespace App\Helpers;

class BackupHelper
{
    /**
     * Format bytes to human readable format
     */
    public static function formatBytes($size, $precision = 2)
    {
        if ($size === 0) return '0 B';
        
        $units = ['B', 'KB', 'MB', 'GB', 'TB', 'PB'];
        $base = log($size, 1024);
        $index = floor($base);
        
        return round(pow(1024, $base - $index), $precision) . ' ' . $units[$index];
    }

    /**
     * Get backup type icon
     */
    public static function getBackupTypeIcon($type)
    {
        $icons = [
            'full' => 'bx-data',
            'database' => 'bx-cylinder',
            'files' => 'bx-folder'
        ];

        return $icons[$type] ?? 'bx-file';
    }

    /**
     * Get backup status color
     */
    public static function getBackupStatusColor($status)
    {
        $colors = [
            'pending' => 'warning',
            'processing' => 'info',
            'completed' => 'success',
            'failed' => 'danger',
            'cancelled' => 'secondary'
        ];

        return $colors[$status] ?? 'secondary';
    }

    /**
     * Calculate estimated backup time based on type and system specs
     */
    public static function estimateBackupTime($type)
    {
        // Get database size
        $dbSize = self::getDatabaseSize();
        
        // Get files size (approximate)
        $filesSize = self::getApplicationFilesSize();

        $estimatedMinutes = 0;

        switch ($type) {
            case 'database':
                // Estimate 1 minute per 100MB of database
                $estimatedMinutes = max(1, ceil($dbSize / (100 * 1024 * 1024)));
                break;
            case 'files':
                // Estimate 2 minutes per 1GB of files
                $estimatedMinutes = max(2, ceil($filesSize / (1024 * 1024 * 1024)) * 2);
                break;
            case 'full':
                // Combine both estimates
                $dbMinutes = max(1, ceil($dbSize / (100 * 1024 * 1024)));
                $filesMinutes = max(2, ceil($filesSize / (1024 * 1024 * 1024)) * 2);
                $estimatedMinutes = $dbMinutes + $filesMinutes;
                break;
        }

        return $estimatedMinutes;
    }

    /**
     * Get database size in bytes
     */
    public static function getDatabaseSize()
    {
        try {
            $database = config('database.connections.mysql.database');
            $result = \DB::select("
                SELECT 
                    ROUND(SUM(data_length + index_length), 0) AS size_bytes
                FROM information_schema.tables 
                WHERE table_schema = ?
            ", [$database]);

            return $result[0]->size_bytes ?? 0;
        } catch (\Exception $e) {
            return 0;
        }
    }

    /**
     * Get application files size (approximate)
     */
    public static function getApplicationFilesSize()
    {
        try {
            $paths = [
                base_path('app'),
                base_path('config'),
                base_path('database'),
                base_path('resources'),
                base_path('routes'),
                base_path('public'),
                storage_path('app')
            ];

            $totalSize = 0;
            foreach ($paths as $path) {
                if (is_dir($path)) {
                    $totalSize += self::getDirectorySize($path);
                }
            }

            return $totalSize;
        } catch (\Exception $e) {
            return 0;
        }
    }

    /**
     * Get directory size recursively
     */
    public static function getDirectorySize($directory)
    {
        $size = 0;
        
        try {
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($iterator as $file) {
                if ($file->isFile()) {
                    $size += $file->getSize();
                }
            }
        } catch (\Exception $e) {
            // Ignore errors and continue
        }

        return $size;
    }

    /**
     * Validate backup file integrity
     */
    public static function validateBackupFile($filePath)
    {
        if (!file_exists($filePath)) {
            return false;
        }

        // Check if it's a valid zip file
        $zip = new \ZipArchive();
        $result = $zip->open($filePath, \ZipArchive::CHECKCONS);
        
        if ($result !== TRUE) {
            return false;
        }

        $zip->close();
        return true;
    }

    /**
     * Get backup retention recommendations
     */
    public static function getRetentionRecommendations()
    {
        return [
            'daily' => [
                'days' => 7,
                'description' => __('Keep daily backups for 1 week')
            ],
            'weekly' => [
                'days' => 30,
                'description' => __('Keep weekly backups for 1 month')
            ],
            'monthly' => [
                'days' => 90,
                'description' => __('Keep monthly backups for 3 months')
            ],
            'yearly' => [
                'days' => 365,
                'description' => __('Keep yearly backups for 1 year')
            ]
        ];
    }

    /**
     * Check if system meets backup requirements
     */
    public static function checkSystemRequirements()
    {
        $requirements = [
            'php_version' => [
                'required' => '8.0',
                'current' => PHP_VERSION,
                'status' => version_compare(PHP_VERSION, '8.0', '>=')
            ],
            'zip_extension' => [
                'required' => true,
                'current' => extension_loaded('zip'),
                'status' => extension_loaded('zip')
            ],
            'mysqldump' => [
                'required' => true,
                'current' => self::checkMysqldump(),
                'status' => self::checkMysqldump()
            ],
            'disk_space' => [
                'required' => '1GB',
                'current' => self::formatBytes(disk_free_space(storage_path())),
                'status' => disk_free_space(storage_path()) > (1024 * 1024 * 1024)
            ]
        ];

        return $requirements;
    }

    /**
     * Check if mysqldump is available
     */
    private static function checkMysqldump()
    {
        exec('mysqldump --version', $output, $returnCode);
        return $returnCode === 0;
    }

    /**
     * Generate backup schedule recommendations
     */
    public static function getScheduleRecommendations()
    {
        return [
            'small_site' => [
                'database' => 'daily',
                'files' => 'weekly',
                'full' => 'monthly'
            ],
            'medium_site' => [
                'database' => 'every_6_hours',
                'files' => 'daily',
                'full' => 'weekly'
            ],
            'large_site' => [
                'database' => 'every_hour',
                'files' => 'every_6_hours',
                'full' => 'daily'
            ]
        ];
    }
}