<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\BackupService;
use App\Models\Backup;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Response;

class BackupController extends Controller
{
    protected $backupService;

    public function __construct(BackupService $backupService)
    {
        $this->middleware('auth');
        $this->middleware('isAdmin');
        $this->backupService = $backupService;
    }

    /**
     * Display backup management interface
     */
    public function index()
    {
        $backups = Backup::orderBy('created_at', 'desc')->paginate(20);
        $diskSpace = $this->backupService->getDiskSpaceInfo();
        $stats = $this->backupService->getBackupStats();
        
        return view('admin.backups.index', compact('backups', 'diskSpace', 'stats'));
    }

    /**
     * Create a new backup
     */
    public function create(Request $request)
    {
        $request->validate([
            'type' => 'required|in:full,database,files',
            'description' => 'nullable|string|max:255'
        ]);

        try {
            $backup = $this->backupService->createBackup(
                $request->type,
                $request->description,
                Auth::id()
            );

            return response()->json([
                'success' => true,
                'message' => __('Backup created successfully'),
                'backup' => $backup
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('Error creating backup: ') . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download a backup file
     */
    public function download($id)
    {
        try {
            $backup = Backup::findOrFail($id);
            
            // Ensure backup disk exists
            $this->ensureBackupDiskExists();
            
            if (!Storage::disk('backups')->exists($backup->file_path)) {
                return redirect()->back()->with('error', __('Backup file not found'));
            }

            $backup->increment('download_count');
            
            return Storage::disk('backups')->download(
                $backup->file_path,
                $backup->original_name
            );
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Error downloading backup: ') . $e->getMessage());
        }
    }

    /**
     * Ensure backup disk configuration exists
     */
    protected function ensureBackupDiskExists()
    {
        if (!config('filesystems.disks.backups')) {
            $backupPath = storage_path('app/backups');
            if (!file_exists($backupPath)) {
                mkdir($backupPath, 0755, true);
            }

            config([
                'filesystems.disks.backups' => [
                    'driver' => 'local',
                    'root' => $backupPath,
                    'url' => env('APP_URL').'/storage/backups',
                    'visibility' => 'private',
                ]
            ]);
        }
    }

    /**
     * Delete a backup
     */
    public function destroy($id)
    {
        try {
            $backup = Backup::findOrFail($id);
            
            // Ensure backup disk exists
            $this->ensureBackupDiskExists();
            
            // Delete file from storage
            if (Storage::disk('backups')->exists($backup->file_path)) {
                Storage::disk('backups')->delete($backup->file_path);
            }
            
            // Delete database record
            $backup->delete();
            
            return response()->json([
                'success' => true,
                'message' => __('Backup deleted successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('Error deleting backup: ') . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Restore from backup
     */
    public function restore(Request $request, $id)
    {
        $request->validate([
            'confirm' => 'required|accepted'
        ]);

        try {
            $backup = Backup::findOrFail($id);
            
            if ($backup->type !== 'database') {
                return response()->json([
                    'success' => false,
                    'message' => __('Only database backups can be restored')
                ], 400);
            }

            $this->backupService->restoreBackup($backup);
            
            return response()->json([
                'success' => true,
                'message' => __('Database restored successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('Error restoring backup: ') . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get backup progress (for AJAX polling)
     */
    public function progress($id)
    {
        $backup = Backup::findOrFail($id);
        
        return response()->json([
            'status' => $backup->status,
            'progress' => $backup->progress,
            'message' => $backup->status_message
        ]);
    }

    /**
     * Cleanup old backups
     */
    public function cleanup(Request $request)
    {
        $request->validate([
            'days' => 'required|integer|min:1|max:365'
        ]);

        try {
            $deletedCount = $this->backupService->cleanupOldBackups($request->days);
            
            return response()->json([
                'success' => true,
                'message' => __('Cleaned up :count old backups', ['count' => $deletedCount])
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('Error during cleanup: ') . $e->getMessage()
            ], 500);
        }
    }
}