<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\BalanceLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class BalanceRequestController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'is_admin']);
    }

    public function index(Request $request)
    {
        $query = BalanceLog::with('user')
            ->where('payment_method', 'bank_transfer')
            ->orderBy('created_at', 'desc');

        // Status filter
        if ($request->status) {
            $query->where('status', $request->status);
        }

        $requests = $query->paginate(20);
        
        $stats = [
            'pending' => BalanceLog::where('payment_method', 'bank_transfer')->where('status', 'pending')->count(),
            'completed' => BalanceLog::where('payment_method', 'bank_transfer')->where('status', 'completed')->count(),
            'rejected' => BalanceLog::where('payment_method', 'bank_transfer')->where('status', 'rejected')->count(),
            'total_amount' => BalanceLog::where('payment_method', 'bank_transfer')->sum('amount'),
            'approved_amount' => BalanceLog::where('payment_method', 'bank_transfer')->where('status', 'completed')->sum('amount'),
        ];

        return view('admin.balance-requests.index', compact('requests', 'stats'));
    }

    public function show($id)
    {
        $request = BalanceLog::with('user')->findOrFail($id);
        return view('admin.balance-requests.show', compact('request'));
    }

    public function approve(Request $request, $id)
    {
        $balanceRequest = BalanceLog::findOrFail($id);
        
        if ($balanceRequest->status !== 'pending') {
            return back()->with('error', 'Bu talep zaten işlenmiş.');
        }

        try {
            // Update balance request status
            $balanceRequest->update([
                'status' => 'completed',
                'approved_by' => auth()->id(),
                'approved_at' => now()
            ]);

            // Add balance to user
            $user = User::find($balanceRequest->user_id);
            $user->increment('balance', $balanceRequest->amount);

            // Create transaction log
            \App\Models\TransactionLog::create([
                'user_id' => $user->id,
                'type' => 'balance_deposit',
                'amount' => $balanceRequest->amount,
                'description' => 'Banka havalesi onaylandı - Admin: ' . auth()->user()->username,
                'status' => 'completed'
            ]);

            return back()->with('success', 'Bakiye yükleme talebi onaylandı ve kullanıcı bakiyesi güncellendi.');

        } catch (\Exception $e) {
            Log::error('Balance request approval error: ' . $e->getMessage());
            return back()->with('error', 'Onaylama işlemi sırasında hata oluştu.');
        }
    }

    public function reject(Request $request, $id)
    {
        $request->validate([
            'rejection_reason' => 'required|string|max:500'
        ]);

        $balanceRequest = BalanceLog::findOrFail($id);
        
        if ($balanceRequest->status !== 'pending') {
            return back()->with('error', 'Bu talep zaten işlenmiş.');
        }

        $balanceRequest->update([
            'status' => 'rejected',
            'rejection_reason' => $request->rejection_reason,
            'rejected_by' => auth()->id(),
            'rejected_at' => now()
        ]);

        return back()->with('success', 'Bakiye yükleme talebi reddedildi.');
    }

    public function downloadReceipt($id)
    {
        $balanceRequest = BalanceLog::findOrFail($id);
        
        if (!$balanceRequest->receipt_path || !Storage::disk('public')->exists($balanceRequest->receipt_path)) {
            return back()->with('error', 'Dekont dosyası bulunamadı.');
        }

        return Storage::disk('public')->download($balanceRequest->receipt_path);
    }
}