<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class LanguageController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('isAdmin');
    }

    /**
     * Get all available languages from JSON files
     */
    private function getAvailableLanguages()
    {
        $langPath = resource_path('lang');
        $languages = [];
        
        if (File::exists($langPath)) {
            $files = File::files($langPath);
            
            foreach ($files as $file) {
                if ($file->getExtension() === 'json') {
                    $code = $file->getFilenameWithoutExtension();
                    $content = json_decode(File::get($file->getPathname()), true);
                    
                    if ($content) {
                        // Check if it has Language Name or create default values
                        $languageName = $content['Language Name'] ?? $this->getDefaultLanguageName($code);
                        $direction = $content['Language Direction'] ?? 'ltr';
                        $icon = $content['Language Icon'] ?? $this->getDefaultIcon($code);
                        $order = (int)($content['Lang Order'] ?? 999);
                        
                        // Calculate completion percentage
                        $totalKeys = count($content) - 5; // Exclude meta keys
                        $filledKeys = 0;
                        $metaKeys = ['Lang Order', 'Language Name', 'Language Code', 'Language Direction', 'Language Icon'];
                        
                        foreach ($content as $key => $value) {
                            if (!in_array($key, $metaKeys) && !empty(trim($value))) {
                                $filledKeys++;
                            }
                        }
                        
                        $completion = $totalKeys > 0 ? round(($filledKeys / $totalKeys) * 100, 1) : 0;
                        
                        $languages[] = [
                            'code' => $code,
                            'name' => $languageName,
                            'native_name' => $languageName,
                            'direction' => $direction,
                            'icon' => $icon,
                            'order' => $order,
                            'file_path' => $file->getPathname(),
                            'key_count' => $totalKeys,
                            'completion' => $completion,
                        ];
                    }
                }
            }
        }
        
        // Sort by order
        usort($languages, function($a, $b) {
            return $a['order'] <=> $b['order'];
        });
        
        return $languages;
    }

    /**
     * Get default language name for code
     */
    private function getDefaultLanguageName($code)
    {
        $defaults = [
            'tr' => 'Türkçe',
            'en' => 'English',
            'ar' => 'العربية',
            'ru' => 'Русский'
        ];
        
        return $defaults[$code] ?? ucfirst($code);
    }

    /**
     * Get default icon for language code
     */
    private function getDefaultIcon($code)
    {
        $defaults = [
            'tr' => '🇹🇷',
            'en' => '🇺🇸',
            'ar' => '🇸🇦',
            'ru' => '🇷🇺'
        ];
        
        return $defaults[$code] ?? '🌐';
    }

    /**
     * Display a listing of languages
     */
    public function index()
    {
        $languages = $this->getAvailableLanguages();
        
        // İstatistikler
        $stats = [
            'total_languages' => count($languages),
            'total_keys' => 0,
            'completion_rate' => 0
        ];
        
        if (!empty($languages)) {
            // En.json'dan toplam anahtar sayısını al
            $enPath = resource_path('lang/en.json');
            if (File::exists($enPath)) {
                $enContent = json_decode(File::get($enPath), true);
                $metaKeys = ['Lang Order', 'Language Name', 'Language Code', 'Language Direction', 'Language Icon'];
                $stats['total_keys'] = count($enContent) - count($metaKeys);
            }
            
            // Ortalama tamamlanma oranı
            $totalCompletion = 0;
            foreach ($languages as $language) {
                $filledKeys = 0;
                $langPath = resource_path('lang/' . $language['code'] . '.json');
                if (File::exists($langPath)) {
                    $content = json_decode(File::get($langPath), true);
                    foreach ($content as $key => $value) {
                        if (!in_array($key, ['Lang Order', 'Language Name', 'Language Code', 'Language Direction', 'Language Icon']) && !empty(trim($value))) {
                            $filledKeys++;
                        }
                    }
                }
                $completion = $stats['total_keys'] > 0 ? ($filledKeys / $stats['total_keys']) * 100 : 0;
                $totalCompletion += $completion;
            }
            $stats['completion_rate'] = $stats['total_languages'] > 0 ? round($totalCompletion / $stats['total_languages'], 1) : 0;
        }
        
        return view('admin.languages.index', compact('languages', 'stats'));
    }

    /**
     * Get languages data for AJAX
     */
    public function getData()
    {
        try {
            $languages = $this->getAvailableLanguages();
            
            return response()->json([
                'success' => true,
                'data' => $languages
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while loading languages')
            ], 500);
        }
    }

    /**
     * Store a newly created language
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'code' => 'required|string|max:10|alpha',
            'name' => 'required|string|max:100',
            'native_name' => 'required|string|max:100',
            'direction' => 'required|in:ltr,rtl',
            'icon' => 'nullable|string|max:10',
            'order' => 'integer|min:1'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $langPath = resource_path('lang');
            $filePath = $langPath . '/' . $request->code . '.json';
            
            // Check if language already exists
            if (File::exists($filePath)) {
                return response()->json([
                    'success' => false,
                    'message' => __('Language already exists')
                ], 422);
            }

            // Get English translations as base
            $enPath = $langPath . '/en.json';
            $baseTranslations = [];
            
            if (File::exists($enPath)) {
                $baseTranslations = json_decode(File::get($enPath), true);
                // Remove meta keys from base
                unset($baseTranslations['Lang Order']);
                unset($baseTranslations['Language Name']);
                unset($baseTranslations['Language Code']);
                unset($baseTranslations['Language Direction']);
                unset($baseTranslations['Language Icon']);
            }

            // Create new language file with meta data
            $newLanguage = [
                'Lang Order' => (string)($request->order ?? 999),
                'Language Name' => $request->native_name,
                'Language Code' => $request->code,
                'Language Direction' => $request->direction,
                'Language Icon' => $request->icon ?? $request->code,
            ];

            // Add base translations (empty values for translation)
            foreach ($baseTranslations as $key => $value) {
                $newLanguage[$key] = ''; // Empty for manual translation
            }

            // Ensure directory exists
            if (!File::exists($langPath)) {
                File::makeDirectory($langPath, 0755, true);
            }

            // Write to file
            File::put($filePath, json_encode($newLanguage, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

            return response()->json([
                'success' => true,
                'message' => __('Language created successfully'),
                'language' => [
                    'code' => $request->code,
                    'name' => $request->name,
                    'native_name' => $request->native_name,
                    'direction' => $request->direction,
                    'icon' => $request->icon ?? $request->code,
                    'order' => $request->order ?? 999,
                    'key_count' => count($baseTranslations)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while creating the language')
            ], 500);
        }
    }

    /**
     * Update language meta information
     */
    public function update(Request $request, $code)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100',
            'native_name' => 'required|string|max:100',
            'direction' => 'required|in:ltr,rtl',
            'icon' => 'nullable|string|max:10',
            'order' => 'integer|min:1'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $langPath = resource_path('lang');
            $filePath = $langPath . '/' . $code . '.json';
            
            if (!File::exists($filePath)) {
                return response()->json([
                    'success' => false,
                    'message' => __('Language file not found')
                ], 404);
            }

            $content = json_decode(File::get($filePath), true);
            
            // Update meta information
            $content['Lang Order'] = (string)($request->order ?? 999);
            $content['Language Name'] = $request->native_name;
            $content['Language Direction'] = $request->direction;
            $content['Language Icon'] = $request->icon ?? $code;

            // Write back to file
            File::put($filePath, json_encode($content, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

            return response()->json([
                'success' => true,
                'message' => __('Language updated successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while updating the language')
            ], 500);
        }
    }

    /**
     * Delete a language
     */
    public function destroy($code)
    {
        try {
            // Prevent deletion of English (base language)
            if ($code === 'en') {
                return response()->json([
                    'success' => false,
                    'message' => __('Cannot delete the base language')
                ], 422);
            }

            $langPath = resource_path('lang');
            $filePath = $langPath . '/' . $code . '.json';
            
            if (!File::exists($filePath)) {
                return response()->json([
                    'success' => false,
                    'message' => __('Language file not found')
                ], 404);
            }

            File::delete($filePath);

            return response()->json([
                'success' => true,
                'message' => __('Language deleted successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while deleting the language')
            ], 500);
        }
    }

    /**
     * Show translations for a language
     */
    public function translations($code)
    {
        $langPath = resource_path('lang');
        $filePath = $langPath . '/' . $code . '.json';
        
        if (!File::exists($filePath)) {
            abort(404, 'Language file not found');
        }

        $content = json_decode(File::get($filePath), true);
        $language = [
            'code' => $code,
            'name' => $content['Language Name'] ?? $code,
            'native_name' => $content['Language Name'] ?? $code,
            'direction' => $content['Language Direction'] ?? 'ltr',
            'icon' => $content['Language Icon'] ?? $code,
            'order' => (int)($content['Lang Order'] ?? 999),
        ];

        return view('admin.languages.translations', compact('language', 'code'));
    }

    /**
     * Get translations data for DataTables
     */
    public function getTranslations(Request $request, $code)
    {
        try {
            $langPath = resource_path('lang');
            $filePath = $langPath . '/' . $code . '.json';
            
            if (!File::exists($filePath)) {
                return response()->json([
                    'success' => false,
                    'message' => __('Language file not found')
                ], 404);
            }

            $content = json_decode(File::get($filePath), true);
            
            if (!$content) {
                return response()->json([
                    'success' => false,
                    'message' => __('Invalid JSON file')
                ], 400);
            }
            
            // Remove meta keys
            $metaKeys = ['Lang Order', 'Language Name', 'Language Code', 'Language Direction', 'Language Icon'];
            foreach ($metaKeys as $metaKey) {
                unset($content[$metaKey]);
            }

            $translations = [];
            foreach ($content as $key => $value) {
                $translations[] = [
                    'key' => $key,
                    'value' => $value,
                    'is_empty' => empty(trim($value)),
                    'character_count' => mb_strlen($value),
                    'word_count' => str_word_count(strip_tags($value))
                ];
            }
            
            // Debug log
            \Log::info('Translations loaded', [
                'code' => $code,
                'total_keys' => count($content),
                'translations_count' => count($translations),
                'request_params' => $request->all()
            ]);

            // Search filter
            if ($request->has('search') && $request->search['value']) {
                $search = strtolower($request->search['value']);
                $translations = array_filter($translations, function($item) use ($search) {
                    return strpos(strtolower($item['key']), $search) !== false || 
                           strpos(strtolower($item['value']), $search) !== false;
                });
            }

            // Status filter (empty/filled)
            if ($request->has('status') && $request->status !== '') {
                if ($request->status === 'empty') {
                    $translations = array_filter($translations, function($item) {
                        return $item['is_empty'] === true;
                    });
                } elseif ($request->status === 'filled') {
                    $translations = array_filter($translations, function($item) {
                        return $item['is_empty'] === false;
                    });
                }
            }

            $totalRecords = count($translations);

            // Sorting
            if ($request->has('order')) {
                $orderColumn = $request->columns[$request->order[0]['column']]['data'];
                $orderDirection = $request->order[0]['dir'];
                
                usort($translations, function($a, $b) use ($orderColumn, $orderDirection) {
                    $result = strcmp($a[$orderColumn], $b[$orderColumn]);
                    return $orderDirection === 'desc' ? -$result : $result;
                });
            }

            // Pagination
            $start = $request->start ?? 0;
            $length = $request->length ?? 25;
            $translations = array_slice($translations, $start, $length);

            return response()->json([
                'draw' => intval($request->draw),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => array_values($translations)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while loading translations')
            ], 500);
        }
    }

    /**
     * Update a translation
     */
    public function updateTranslation(Request $request, $code)
    {
        $validator = Validator::make($request->all(), [
            'key' => 'required|string',
            'value' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $langPath = resource_path('lang');
            $filePath = $langPath . '/' . $code . '.json';
            
            if (!File::exists($filePath)) {
                return response()->json([
                    'success' => false,
                    'message' => __('Language file not found')
                ], 404);
            }

            $content = json_decode(File::get($filePath), true);
            
            // Update translation
            $content[$request->key] = $request->value;

            // Write back to file
            File::put($filePath, json_encode($content, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

            return response()->json([
                'success' => true,
                'message' => __('Translation updated successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while updating the translation')
            ], 500);
        }
    }

    /**
     * Add new translation key
     */
    public function addTranslation(Request $request, $code)
    {
        $validator = Validator::make($request->all(), [
            'key' => 'required|string',
            'value' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $langPath = resource_path('lang');
            $filePath = $langPath . '/' . $code . '.json';
            
            if (!File::exists($filePath)) {
                return response()->json([
                    'success' => false,
                    'message' => __('Language file not found')
                ], 404);
            }

            $content = json_decode(File::get($filePath), true);
            
            // Check if key already exists
            if (isset($content[$request->key])) {
                return response()->json([
                    'success' => false,
                    'message' => __('Translation key already exists')
                ], 422);
            }

            // Add new translation
            $content[$request->key] = $request->value;

            // Write back to file
            File::put($filePath, json_encode($content, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

            return response()->json([
                'success' => true,
                'message' => __('Translation added successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while adding the translation')
            ], 500);
        }
    }

    /**
     * Export language file
     */
    public function export($code)
    {
        try {
            $langPath = resource_path('lang');
            $filePath = $langPath . '/' . $code . '.json';
            
            if (!File::exists($filePath)) {
                return response()->json([
                    'success' => false,
                    'message' => __('Language file not found')
                ], 404);
            }

            $content = File::get($filePath);
            $filename = $code . '.json';

            return response($content)
                ->header('Content-Type', 'application/json')
                ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred during export')
            ], 500);
        }
    }
}