<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WebsiteVisitTask;
use App\Models\WebsiteVisit;
use App\Models\WebsiteVisitProof;
use App\Models\Country;
use App\Models\City;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;

class WebsiteVisitTaskController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('isAdmin');
    }

    /**
     * Display all website visit tasks for admin
     */
    public function index()
    {
        $tasks = WebsiteVisitTask::with(['country', 'city', 'creator'])
            ->withCount(['visits', 'completedVisits'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $settings = app('settings');

        return view('admin.website-visits.index', compact('tasks', 'settings'));
    }

    /**
     * Show task statistics and details
     */
    public function show(WebsiteVisitTask $task)
    {
        $task->load(['country', 'city', 'creator']);
        
        $stats = [
            'total_visits' => $task->visits()->count(),
            'completed_visits' => $task->completedVisits()->count(),
            'failed_visits' => $task->visits()->where('status', 'failed')->count(),
            'total_rewards_given' => $task->visits()->where('reward_given', true)->count() * $task->reward_amount,
            'unique_users' => $task->visits()->distinct('user_id')->count(),
            'avg_duration' => $task->completedVisits()->avg('duration'),
            'today_visits' => $task->visits()->whereDate('created_at', today())->count(),
            'pending_proofs' => WebsiteVisitProof::whereHas('visit', function($q) use ($task) {
                $q->where('task_id', $task->id);
            })->where('status', 0)->count(),
        ];

        $recentVisits = $task->visits()
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        return view('admin.website-visits.show', compact('task', 'stats', 'recentVisits'));
    }

    /**
     * Get recent visits data via AJAX
     */
    public function getRecentVisits(WebsiteVisitTask $task, Request $request)
    {
        $query = $task->visits()->with(['user', 'proofs']);
        
        // Apply status filter if provided
        if ($request->has('status') && !empty($request->status)) {
            // Handle proof-based status filtering
            if ($request->status === 'completed') {
                $query->where(function($q) {
                    $q->where('status', 'completed')
                      ->orWhereHas('proofs', function($proofQuery) {
                          $proofQuery->where('status', 1); // Approved proofs
                      });
                });
            } else {
                $query->where('status', $request->status);
            }
        }
        
        $recentVisits = $query->orderBy('created_at', 'desc')->limit(5)->get();
        
        $html = '';
        
        if ($recentVisits->count() > 0) {
            foreach ($recentVisits as $visit) {
                // Check for approved proof
                $hasApprovedProof = $visit->proofs()->where('status', 1)->exists();
                $displayStatus = $hasApprovedProof ? 'completed' : $visit->status;
                
                // Determine completed date
                $approvedProof = $visit->proofs()->where('status', 1)->first();
                $completedDate = null;
                
                if ($approvedProof && $approvedProof->reviewed_at) {
                    $completedDate = $approvedProof->reviewed_at;
                } elseif ($visit->completed_at) {
                    $completedDate = $visit->completed_at;
                } elseif ($visit->end_time && $visit->status == 'completed') {
                    $completedDate = $visit->end_time;
                }
                
                $html .= '<tr>';
                
                // User column
                $html .= '<td>';
                $html .= '<div class="d-flex align-items-center">';
                $html .= '<img src="' . asset('assets/images/avatars/' . ($visit->user->avatar ?? 'avatar-1.png')) . '" ';
                $html .= 'alt="' . ($visit->user->account_holder ?? $visit->user->name) . '" class="rounded-circle" width="32" height="32">';
                $html .= '<div class="ms-2">';
                $html .= '<a href="' . route('user.profile') . '?user=' . $visit->user->id . '" class="text-decoration-none">';
                $html .= '<h6 class="mb-0 text-primary">' . ($visit->user->account_holder ?? $visit->user->name) . '</h6>';
                $html .= '</a>';
                $html .= '<p class="text-muted mb-0 small">' . $visit->user->email . '</p>';
                $html .= '</div></div></td>';
                
                // Status column
                $html .= '<td>';
                if ($displayStatus == 'completed') {
                    $html .= '<span class="badge bg-success">' . __('Completed') . '</span>';
                    if ($hasApprovedProof && $visit->status != 'completed') {
                        $html .= '<small class="text-muted d-block">' . __('Proof Approved') . '</small>';
                    }
                } elseif ($displayStatus == 'failed') {
                    $html .= '<span class="badge bg-danger">' . __('Failed') . '</span>';
                } elseif ($displayStatus == 'in_progress') {
                    $html .= '<span class="badge bg-warning">' . __('In Progress') . '</span>';
                    if ($visit->proofs()->where('status', 0)->exists()) {
                        $html .= '<small class="text-info d-block">' . __('Proof Pending') . '</small>';
                    }
                } else {
                    $html .= '<span class="badge bg-secondary">' . __('Started') . '</span>';
                }
                $html .= '</td>';
                
                // Duration column
                $html .= '<td>';
                if ($visit->duration) {
                    $html .= $visit->duration . 's';
                } else {
                    $html .= '<span class="text-muted">-</span>';
                }
                $html .= '</td>';
                
                // Reward given column
                $html .= '<td>';
                if ($visit->reward_given) {
                    $html .= '<span class="badge bg-success">' . __('Yes') . '</span>';
                } else {
                    $html .= '<span class="badge bg-secondary">' . __('No') . '</span>';
                }
                $html .= '</td>';
                
                // Started at column
                $html .= '<td>' . $visit->created_at->format('d.m.Y H:i') . '</td>';
                
                // Completed at column
                $html .= '<td>';
                if ($completedDate) {
                    $html .= $completedDate->format('d.m.Y H:i');
                    if ($approvedProof && $approvedProof->reviewed_at) {
                        $html .= '<small class="text-success d-block">' . __('Proof Approved') . '</small>';
                    }
                } else {
                    $html .= '<span class="text-muted">-</span>';
                }
                $html .= '</td>';
                
                $html .= '</tr>';
            }
        } else {
            $html = '<tr><td colspan="6" class="text-center">' . __('No visits found') . '</td></tr>';
        }
        
        // Also get updated stats
        $stats = [
            'total_visits' => $task->visits()->count(),
            'completed_visits' => $task->completedVisits()->count(),
            'failed_visits' => $task->visits()->where('status', 'failed')->count(),
            'unique_users' => $task->visits()->distinct('user_id')->count(),
            'today_visits' => $task->visits()->whereDate('created_at', today())->count(),
            'pending_proofs' => \App\Models\WebsiteVisitProof::whereHas('visit', function($q) use ($task) {
                $q->where('task_id', $task->id);
            })->where('status', 0)->count(),
        ];
        
        return response()->json([
            'success' => true,
            'html' => $html,
            'stats' => $stats
        ]);
    }

    /**
     * Get website visit proofs for admin review
     */
    public function proofs()
    {
        $proofs = WebsiteVisitProof::with(['visit.task', 'user', 'files'])
            ->pending()
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.website-visits.proofs.index', compact('proofs'));
    }

    /**
     * Approve website visit proof
     */
    public function approveProof(Request $request, $proofId)
    {
        $proof = WebsiteVisitProof::findOrFail($proofId);
        $visit = $proof->visit;
        $user = $visit->user;

        // Approve proof
        $proof->update([
            'status' => 1,
            'reviewed_by' => Auth::id(),
            'reviewed_at' => now()
        ]);

        // Give reward if not already given
        if (!$visit->reward_given) {
            $rewardAmount = $visit->task->reward_amount;
            $user->balance += $rewardAmount;
            $user->save();

            // Log balance change
            \App\Models\BalanceLog::create([
                'userId' => $user->id,
                'TaskId' => $visit->id,
                'balance' => $user->balance,
                'balance_before' => $user->balance - $rewardAmount,
                'added_user' => Auth::id()
            ]);

            $visit->update(['reward_given' => true]);
        }

        return response()->json([
            'success' => true,
            'message' => __('Proof approved and reward given to user.')
        ]);
    }

    /**
     * Deny website visit proof
     */
    public function denyProof(Request $request, $proofId)
    {
        $request->validate([
            'reason' => 'required|string|max:500'
        ]);

        $proof = WebsiteVisitProof::findOrFail($proofId);
        
        $proof->update([
            'status' => 2,
            'reason' => $request->reason,
            'reviewed_by' => Auth::id(),
            'reviewed_at' => now()
        ]);

        return response()->json([
            'success' => true,
            'message' => __('Proof denied.')
        ]);
    }

    /**
     * Export website visits for a specific task as Excel
     */
    public function exportVisits(WebsiteVisitTask $task, Request $request)
    {
        $query = $task->visits()->with(['user', 'proofs']);
        
        // Apply status filter if provided
        if ($request->has('status') && !empty($request->status)) {
            $query->where('status', $request->status);
        }
        
        $visits = $query->orderBy('created_at', 'desc')->get();
        
        $filename = 'website_visits_task_' . $task->id . '_' . date('Y-m-d_H-i-s') . '.pdf';
        
        // Create Excel data array
        $excelData = [];
        
        // Headers
        $excelData[] = [
            __('Task Title'),
            __('User'),
            __('Email'),
            __('Status'),
            __('Duration') . ' (' . __('seconds') . ')',
            __('Reward Given'),
            __('Started At'),
            __('Completed At'),
            __('IP Address'),
            __('Proof Status'),
            __('Proof Count'),
            __('Reward Amount')
        ];
        
        foreach ($visits as $visit) {
            $proofStatus = '';
            $proofCount = $visit->proofs->count();
            
            if ($proofCount > 0) {
                $approvedProofs = $visit->proofs->where('status', 1)->count();
                $pendingProofs = $visit->proofs->where('status', 0)->count();
                $deniedProofs = $visit->proofs->where('status', 2)->count();
                
                if ($approvedProofs > 0) {
                    $proofStatus = __('Approved');
                } elseif ($pendingProofs > 0) {
                    $proofStatus = __('Pending');
                } elseif ($deniedProofs > 0) {
                    $proofStatus = __('Denied');
                }
            } else {
                $proofStatus = __('No Proof');
            }
            
            // Status translation
            $statusText = '';
            switch ($visit->status) {
                case 'completed':
                    $statusText = __('Completed');
                    break;
                case 'failed':
                    $statusText = __('Failed');
                    break;
                case 'in_progress':
                    $statusText = __('In Progress');
                    break;
                default:
                    $statusText = __('Started');
                    break;
            }
            
            $excelData[] = [
                $task->title,
                $visit->user->account_holder ?? $visit->user->name,
                $visit->user->email,
                $statusText,
                $visit->duration ?? 0,
                $visit->reward_given ? __('Yes') : __('No'),
                $visit->created_at->format('d.m.Y H:i'),
                $visit->completed_at ? $visit->completed_at->format('d.m.Y H:i') : '-',
                $visit->ip_address ?? '-',
                $proofStatus,
                $proofCount,
                number_format($task->reward_amount, 2, ',', '.') . (app('settings')['currency'] ?? 'TL')
            ];
        }
        
        return $this->generatePdfFile($excelData, $filename, $task);
    }
    
    /**
     * Generate PDF file using DomPDF
     */
    private function generatePdfFile($data, $filename, $task)
    {
        // Generate HTML content for PDF
        $html = $this->generatePdfHtml($data, $task);
        
        // Create PDF using DomPDF
        $pdf = Pdf::loadHTML($html);
        
        // Set PDF options
        $pdf->setPaper('A4', 'landscape'); // Landscape for better table viewing
        $pdf->setOptions([
            'defaultFont' => 'DejaVu Sans',
            'isHtml5ParserEnabled' => true,
            'isPhpEnabled' => true,
            'isRemoteEnabled' => true,
        ]);
        
        // Return PDF download response
        return $pdf->download($filename);
    }
    
    /**
     * Generate HTML content optimized for PDF
     */
    private function generatePdfHtml($data, $task)
    {
        $html = '<!DOCTYPE html>';
        $html .= '<html><head>';
        $html .= '<meta charset="UTF-8">';
        $html .= '<title>' . htmlspecialchars($task->title . ' - ' . __('Website Visits Report')) . '</title>';
        $html .= '<style>';
        
        // PDF-optimized CSS
        $html .= 'body { font-family: DejaVu Sans, sans-serif; margin: 15px; color: #333; font-size: 10px; }';
        $html .= 'h1 { color: #2c3e50; text-align: center; margin-bottom: 20px; font-size: 18px; }';
        $html .= '.header-info { margin-bottom: 20px; background: #f8f9fa; padding: 10px; border: 1px solid #ddd; }';
        $html .= '.header-info p { margin: 4px 0; font-size: 9px; }';
        
        // Table styles optimized for PDF
        $html .= 'table { width: 100%; border-collapse: collapse; margin-top: 15px; font-size: 8px; }';
        $html .= 'th { background-color: #4472C4; color: white; padding: 6px 4px; border: 1px solid #000; font-weight: bold; text-align: left; }';
        $html .= 'td { padding: 5px 4px; border: 1px solid #666; vertical-align: top; word-wrap: break-word; }';
        $html .= 'tr:nth-child(even) { background-color: #f9f9f9; }';
        
        // Footer styles
        $html .= '.footer { margin-top: 20px; text-align: center; font-size: 8px; color: #666; border-top: 1px solid #ddd; padding-top: 10px; }';
        
        $html .= '</style>';
        $html .= '</head><body>';
        
        // Header information
        $html .= '<h1>' . htmlspecialchars($task->title . ' - ' . __('Website Visits Report')) . '</h1>';
        $html .= '<div class="header-info">';
        $html .= '<p><strong>' . __('Task Title') . ':</strong> ' . htmlspecialchars($task->title) . '</p>';
        $html .= '<p><strong>' . __('Website URL') . ':</strong> ' . htmlspecialchars($task->website_url) . '</p>';
        $html .= '<p><strong>' . __('Generated') . ':</strong> ' . date('d.m.Y H:i') . '</p>';
        $html .= '<p><strong>' . __('Total Records') . ':</strong> ' . (count($data) - 1) . '</p>';
        $html .= '</div>';
        
        // Table
        $html .= '<table>';
        
        foreach ($data as $index => $row) {
            if ($index === 0) {
                $html .= '<thead><tr>';
                foreach ($row as $cell) {
                    $html .= '<th>' . htmlspecialchars($cell) . '</th>';
                }
                $html .= '</tr></thead><tbody>';
            } else {
                $html .= '<tr>';
                foreach ($row as $cell) {
                    $html .= '<td>' . htmlspecialchars($cell) . '</td>';
                }
                $html .= '</tr>';
            }
        }
        
        $html .= '</tbody></table>';
        
        // Footer
        $html .= '<div class="footer">';
        $html .= '<p>' . __('Generated by') . ' ' . config('app.name', 'Laravel') . ' - ' . date('d.m.Y H:i') . '</p>';
        $html .= '</div>';
        
        $html .= '</body></html>';
        
        return $html;
    }
    

    /**
     * Delete a website visit task (admin only)
     */
    public function destroy(WebsiteVisitTask $task)
    {
        try {
            $task->delete();
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => __('Website visit task deleted successfully.')
                ]);
            }
            
            return redirect()->route('admin.website-visits.index')
                ->with('success', __('Website visit task deleted successfully.'));
        } catch (\Exception $e) {
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => __('Error deleting task.')
                ], 500);
            }
            
            return redirect()->route('admin.website-visits.index')
                ->with('error', __('Error deleting task.'));
        }
    }
}