<?php

namespace App\Http\Controllers;

use App\Models\Announcement;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class AnnouncementController extends Controller
{
    // User: Duyuru listesi
    public function index(Request $request)
    {
        $user = Auth::user();
        
        if ($user->user_type == 1) {
            // Admin view
            return $this->adminIndex($request);
        }
        
        // User view
        $announcements = Announcement::active()
            ->forUser($user->id)
            ->with(['creator'])
            ->orderBy('priority', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return view('announcements.index', compact('announcements'));
    }
    
    // Admin: Duyuru yönetimi
    public function adminIndex(Request $request)
    {
        $query = Announcement::with(['creator']);
        
        // Filters
        if ($request->type) {
            $query->where('type', $request->type);
        }
        
        if ($request->priority) {
            $query->where('priority', $request->priority);
        }
        
        if ($request->status === 'active') {
            $query->active();
        } elseif ($request->status === 'inactive') {
            $query->where('is_active', false);
        } elseif ($request->status === 'expired') {
            $query->where('expires_at', '<', now());
        }
        
        $announcements = $query->orderBy('created_at', 'desc')->paginate(20);
        
        // Statistics
        $stats = [
            'total' => Announcement::count(),
            'active' => Announcement::active()->count(),
            'general' => Announcement::where('type', 'general')->count(),
            'user_specific' => Announcement::where('type', 'user_specific')->count(),
            'urgent' => Announcement::where('priority', 'urgent')->active()->count()
        ];
        
        return view('announcements.admin.index', compact('announcements', 'stats'));
    }
    
    // Duyuru oluşturma formu (Admin)
    public function create()
    {
        $this->authorize('create', Announcement::class);
        return view('announcements.create');
    }
    
    // Duyuru oluştur (Admin)
    public function store(Request $request)
    {
        $this->authorize('create', Announcement::class);
        
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string|max:5000',
            'type' => 'required|in:general,user_specific',
            'priority' => 'required|in:low,medium,high,urgent',
            'show_as_popup' => 'boolean',
            'expires_at' => 'nullable|date|after:now',
            'target_users' => 'required_if:type,user_specific|array',
            'target_users.*' => 'exists:users,id'
        ]);
        
        DB::beginTransaction();
        try {
            $announcement = Announcement::create([
                'title' => $request->title,
                'content' => $request->content,
                'type' => $request->type,
                'priority' => $request->priority,
                'show_as_popup' => $request->show_as_popup ?? true,
                'expires_at' => $request->expires_at,
                'created_by' => Auth::id()
            ]);
            
            // Kullanıcı özel duyuru ise hedef kullanıcıları ekle
            if ($request->type === 'user_specific' && $request->target_users) {
                $announcement->targetUsers()->attach($request->target_users);
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => __('Announcement created successfully'),
                'announcement_id' => $announcement->id
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('Error creating announcement')
            ], 500);
        }
    }
    
    // Duyuru detayı
    public function show($id)
    {
        $announcement = Announcement::with(['creator', 'targetUsers'])->findOrFail($id);
        $user = Auth::user();
        
        // Check permissions
        if ($user->user_type != 1 && !$announcement->canBeSeenByUser($user->id)) {
            abort(403);
        }
        
        // Mark as read for user
        if ($user->user_type != 1) {
            $announcement->markAsReadByUser($user->id);
        }
        
        return view('announcements.show', compact('announcement'));
    }
    
    // Duyuru düzenle (Admin)
    public function edit($id)
    {
        $announcement = Announcement::with(['targetUsers'])->findOrFail($id);
        $this->authorize('update', $announcement);
        
        return view('announcements.edit', compact('announcement'));
    }
    
    // Duyuru güncelle (Admin)
    public function update(Request $request, $id)
    {
        $announcement = Announcement::findOrFail($id);
        $this->authorize('update', $announcement);
        
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string|max:5000',
            'type' => 'required|in:general,user_specific',
            'priority' => 'required|in:low,medium,high,urgent',
            'is_active' => 'boolean',
            'show_as_popup' => 'boolean',
            'expires_at' => 'nullable|date',
            'target_users' => 'required_if:type,user_specific|array',
            'target_users.*' => 'exists:users,id'
        ]);
        
        DB::beginTransaction();
        try {
            $announcement->update([
                'title' => $request->title,
                'content' => $request->content,
                'type' => $request->type,
                'priority' => $request->priority,
                'is_active' => $request->is_active ?? true,
                'show_as_popup' => $request->show_as_popup ?? true,
                'expires_at' => $request->expires_at
            ]);
            
            // Update target users
            if ($request->type === 'user_specific' && $request->target_users) {
                $announcement->targetUsers()->sync($request->target_users);
            } else {
                $announcement->targetUsers()->detach();
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => __('Announcement updated successfully')
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('Error updating announcement')
            ], 500);
        }
    }
    
    // Duyuru sil (Admin)
    public function destroy($id)
    {
        $announcement = Announcement::findOrFail($id);
        $this->authorize('delete', $announcement);
        
        $announcement->delete();
        
        return response()->json([
            'success' => true,
            'message' => __('Announcement deleted successfully')
        ]);
    }
    
    // Duyuru okundu olarak işaretle
    public function markAsRead($id)
    {
        $announcement = Announcement::findOrFail($id);
        $user = Auth::user();
        
        if ($announcement->canBeSeenByUser($user->id)) {
            $announcement->markAsReadByUser($user->id);
        }
        
        return response()->json([
            'success' => true,
            'message' => __('Announcement marked as read')
        ]);
    }
    
    // Popup duyuruları getir
    public function getPopupAnnouncements()
    {
        $user = Auth::user();
        
        $announcements = Announcement::active()
            ->forUser($user->id)
            ->unreadByUser($user->id)
            ->where('show_as_popup', true)
            ->orderBy('priority', 'desc')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
            
        return response()->json([
            'success' => true,
            'announcements' => $announcements->map(function($announcement) {
                return [
                    'id' => $announcement->id,
                    'title' => $announcement->title,
                    'content' => $announcement->formatted_content,
                    'priority' => $announcement->priority,
                    'priority_badge' => $announcement->priority_badge,
                    'created_at' => $announcement->created_at->format('d.m.Y H:i')
                ];
            })
        ]);
    }
    
    // DataTable API (Admin)
    public function adminDataTable(Request $request)
    {
        try {
            $query = Announcement::with(['creator', 'readByUsers', 'targetUsers']);
            
            // Filters
            if ($request->type) {
                $query->where('type', $request->type);
            }
            
            if ($request->priority) {
                $query->where('priority', $request->priority);
            }
            
            if ($request->status === 'active') {
                $query->where('is_active', true)
                      ->where(function($q) {
                          $q->whereNull('expires_at')
                            ->orWhere('expires_at', '>', now());
                      });
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            } elseif ($request->status === 'expired') {
                $query->where('expires_at', '<', now());
            }
            
            // Search
            if ($request->search && isset($request->search['value']) && $request->search['value']) {
                $searchValue = $request->search['value'];
                $query->where(function($q) use ($searchValue) {
                    $q->where('title', 'like', "%{$searchValue}%")
                      ->orWhere('content', 'like', "%{$searchValue}%");
                });
            }
            
            $totalRecords = $query->count();
            
            // Ordering
            if ($request->order && isset($request->order[0])) {
                $orderColumnIndex = $request->order[0]['column'];
                $orderDirection = $request->order[0]['dir'];
                
                if (isset($request->columns[$orderColumnIndex]['name'])) {
                    $orderColumn = $request->columns[$orderColumnIndex]['name'];
                    
                    // Handle special ordering cases
                    if ($orderColumn === 'read_stats') {
                        // Skip ordering for read stats as it's calculated
                        $query->orderBy('created_at', 'desc');
                    } else {
                        $query->orderBy($orderColumn, $orderDirection);
                    }
                } else {
                    $query->orderBy('created_at', 'desc');
                }
            } else {
                $query->orderBy('created_at', 'desc');
            }
            
            // Pagination
            $start = $request->start ?? 0;
            $length = $request->length ?? 25;
            $announcements = $query->skip($start)->take($length)->get();
            
            $data = $announcements->map(function($announcement) {
                $readCount = $announcement->readByUsers->count();
                $targetUsersCount = $announcement->type === 'user_specific' ? $announcement->targetUsers->count() : 0;
                
                return [
                    'id' => $announcement->id,
                    'title' => $announcement->title,
                    'content_preview' => Str::limit(strip_tags($announcement->content), 60),
                    'type' => $announcement->type,
                    'type_text' => $announcement->type_text,
                    'priority' => $announcement->priority,
                    'priority_text' => $announcement->priority_text,
                    'priority_badge' => $announcement->priority_badge,
                    'is_active' => $announcement->is_active,
                    'is_expired' => $announcement->isExpired(),
                    'status_text' => $announcement->is_active && !$announcement->isExpired() ? __('Active') : 
                                   ($announcement->isExpired() ? __('Expired') : __('Inactive')),
                    'status_badge' => $announcement->is_active && !$announcement->isExpired() ? 'success' : 
                                    ($announcement->isExpired() ? 'warning' : 'secondary'),
                    'show_as_popup' => $announcement->show_as_popup,
                    'read_count' => $readCount,
                    'target_users_count' => $targetUsersCount,
                    'created_at' => $announcement->created_at->format('d.m.Y H:i'),
                    'creator_name' => $announcement->creator->account_holder ?? $announcement->creator->name
                ];
            });
            
            return response()->json([
                'draw' => intval($request->draw ?? 1),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => $data
            ]);
            
        } catch (\Exception $e) {
            Log::error('AdminDataTable error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'draw' => intval($request->draw ?? 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading data: ' . $e->getMessage()
            ], 500);
        }
    }
    
    // Edit form data (Ajax)
    public function getEditData($id)
    {
        try {
            $announcement = Announcement::with(['targetUsers'])->findOrFail($id);
            $this->authorize('update', $announcement);
            
            return response()->json([
                'success' => true,
                'announcement' => [
                    'id' => $announcement->id,
                    'title' => $announcement->title,
                    'content' => $announcement->content,
                    'type' => $announcement->type,
                    'priority' => $announcement->priority,
                    'is_active' => $announcement->is_active,
                    'show_as_popup' => $announcement->show_as_popup,
                    'expires_at' => $announcement->expires_at ? $announcement->expires_at->format('Y-m-d\TH:i') : null,
                    'target_users' => $announcement->targetUsers->map(function($user) {
                        return [
                            'id' => $user->id,
                            'account_holder' => $user->account_holder,
                            'email' => $user->email
                        ];
                    })
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('Error loading announcement data')
            ], 500);
        }
    }
}