<?php

namespace App\Http\Controllers\Api; // Keep this one

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\WatchTask;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class WatchTaskController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
   public function index()
{
    $userId = auth()->id();
    $tasks = WatchTask::where('created_by', '!=', $userId)
                ->latest()
                ->get()
                ->map(function($task) use ($userId) {
                    return [
                        'id'                  => $task->id,
                        'title'               => $task->title,
                        'description'         => $task->description,
                        'youtube_url'         => $task->youtube_url,
                        'youtube_video_id'    => $task->youtube_video_id,
                        'formatted_duration'  => $task->formatted_duration,
                        'reward_amount'       => $task->reward_amount,
                        'max_completions'     => $task->max_completions,
                        'current_completions' => $task->current_completions,
                        'is_active'           => $task->is_active,
                        'created_by'          => $task->created_by,
                        // Aşağıdakileri ekliyoruz:
                        'is_completed'        => $task->isCompletedByUser($userId),
                        'can_watch'           => $task->canBeCompleted() && !$task->isCompletedByUser($userId),
                        'can_manage'          => $task->created_by == $userId || auth()->user()->isAdmin(),
                    ];
                });

    return response()->json([
        'success' => true,
        'tasks'   => $tasks
    ]);
}


   public function store(Request $request)
    {
        $data = $request->validate([
            'title'         => 'required|string|max:255',
            'description'   => 'nullable|string',
            'youtube_url'   => 'required|url',
            'watch_duration'=> 'required|integer|min:10|max:3600',
            'reward_amount' => 'required|numeric|min:0.01',
            'max_completions'=> 'required|integer|min:1',
        ]);

        // Video ID'yi URL'den ayıkla
        $videoId = WatchTask::extractVideoId($data['youtube_url']);
        if (! $videoId) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz YouTube URL\'si.'
            ], 422);
        }

        // Toplam maliyeti hesapla
        $totalCost = $data['reward_amount'] * $data['max_completions'];
        $user = auth()->user();

        // Bakiye kontrolü
        if ($user->balance < $totalCost) {
            return response()->json([
                'success' => false,
                'message' => 'Yetersiz bakiye. Gerekli miktar: ' . $totalCost . '₺, Mevcut bakiye: ' . $user->balance . '₺'
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Kullanıcının bakiyesinden düş
            $user->balance -= $totalCost;
            $user->save();

            // Gerekli tüm alanları birleştir
            $taskData = array_merge($data, [
                'youtube_video_id' => $videoId,
                'created_by'       => auth()->id(),
            ]);

            $task = WatchTask::create($taskData);

            // Balance log kaydı
            \App\Models\BalanceLog::create([
                'userId' => $user->id,
                'balance_before' => $user->balance + $totalCost,
                'balance' => -$totalCost,
                'balance_after' => $user->balance,
                'description' => "Watch task oluşturma: {$data['title']}",
                'type' => 'watch_task_creation',
                'reference_type' => 'watch_task',
                'reference_id' => $task->id,
                'created_at' => now(),
                'updated_at' => now()
            ]);

            DB::commit();

            return response()->json([
                'success'     => true,
                'message'     => 'Görev başarıyla oluşturuldu. Toplam maliyet: ' . $totalCost . '₺',
                'task'        => $task,
                'new_balance' => (float) number_format($user->balance, 2, '.', ''),
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Görev oluşturulurken bir hata oluştu.'
            ], 500);
        }
    }

public function toggleStatus(WatchTask $task)
{
    // Sahip değil ve admin değilse erişimi yasakla
    if ($task->created_by !== auth()->id() && ! auth()->user()->isAdmin()) {
        abort(403, 'Bu görevi yönetme yetkiniz yok.');
    }

    $task->is_active = ! $task->is_active;
    $task->save();

    return response()->json([
        'success' => true,
        'message' => 'Durum güncellendi'
    ]);
}

 public function destroy(WatchTask $task)
    {
        // Yetki kontrolü
        if ($task->created_by !== auth()->id() && ! auth()->user()->isAdmin()) {
            return response()->json([
                'success' => false,
                'message' => 'Bu işlem için yetkiniz yok.'
            ], 403);
        }

        try {
            // Eğer tamamlanma kaydı varsa, silme yerine pasife al
            if ($task->completions()->exists()) {
                $task->is_active = false;
                $task->save();

                return response()->json([
                    'success'       => true,
                    'message'       => 'Bu görevde tamamlanma kaydı bulunduğu için silinmedi; pasif yapıldı.',
                    'refund_amount' => 0,
                ]);
            }

            // Aksi halde gerçekten sil ve varsa bakiye iadesi yap
            $refundAmount = 0;

            DB::transaction(function () use ($task, &$refundAmount) {
                // Kalan kullanıcılar için iade hesapla
                if ($task->current_completions < $task->max_completions) {
                    $remaining    = $task->max_completions - $task->current_completions;
                    $refundAmount = $remaining * $task->reward_amount;

                    // Oluşturan kullanıcıya iade
                    $creator = $task->creator;
                    $creator->balance += $refundAmount;
                    $creator->save();

                    \App\Models\BalanceLog::create([
                        'userId'         => $creator->id,
                        'balance_before' => $creator->balance - $refundAmount,
                        'balance'        => $refundAmount,
                        'balance_after'  => $creator->balance,
                        'description'    => "Watch task silme iadesi: {$task->title}",
                        'type'           => 'watch_task_refund',
                        'reference_type' => 'watch_task',
                        'reference_id'   => $task->id,
                        'created_at'     => now(),
                        'updated_at'     => now(),
                    ]);
                }

                // Görevi sil
                $task->delete();
            });

            return response()->json([
                'success'       => true,
                'message'       => 'Görev silindi' . ($refundAmount > 0 ? " ve {$refundAmount}₺ iade edildi." : '.'),
                'refund_amount' => $refundAmount,
            ]);
        } catch (\Exception $e) {
            Log::error("Error deleting WatchTask #{$task->id}: " . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Silme işlemi sırasında bir hata oluştu.'
            ], 500);
        }
    }

    // Kullanıcının kendi görevlerini listele
    public function userTasks()
    {
        $userId = auth()->id();
        $tasks = WatchTask::where('created_by', $userId)
                    ->latest()
                    ->get()
                    ->map(function($task) {
                        return [
                            'id'                  => $task->id,
                            'title'               => $task->title,
                            'description'         => $task->description,
                            'youtube_url'         => $task->youtube_url,
                            'youtube_video_id'    => $task->youtube_video_id,
                            'watch_duration'      => $task->watch_duration,
                            'formatted_duration'  => $task->formatted_duration,
                            'reward_amount'       => $task->reward_amount,
                            'max_completions'     => $task->max_completions,
                            'current_completions' => $task->current_completions,
                            'is_active'           => $task->is_active,
                            'created_at'          => $task->created_at->format('d.m.Y H:i'),
                        ];
                    });

        return response()->json([
            'success' => true,
            'tasks'   => $tasks
        ]);
    }

    // Tek görev detayı
    public function show(WatchTask $task)
    {
        // Yetki kontrolü
        if ($task->created_by != auth()->id() && !auth()->user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Bu işlem için yetkiniz yok.'], 403);
        }

        return response()->json([
            'success' => true,
            'task' => [
                'id'                  => $task->id,
                'title'               => $task->title,
                'description'         => $task->description,
                'youtube_url'         => $task->youtube_url,
                'youtube_video_id'    => $task->youtube_video_id,
                'watch_duration'      => $task->watch_duration,
                'formatted_duration'  => $task->formatted_duration,
                'reward_amount'       => $task->reward_amount,
                'max_completions'     => $task->max_completions,
                'current_completions' => $task->current_completions,
                'is_active'           => $task->is_active,
                'created_at'          => $task->created_at->format('d.m.Y H:i'),
            ]
        ]);
    }

    // Görev güncelle
    public function update(Request $request, WatchTask $task)
    {
        // Yetki kontrolü
        if ($task->created_by != auth()->id() && !auth()->user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Bu işlem için yetkiniz yok.'], 403);
        }

        $data = $request->validate([
            'title'         => 'required|string|max:255',
            'description'   => 'nullable|string',
            'watch_duration'=> 'required|integer|min:10|max:3600',
            'reward_amount' => 'required|numeric|min:0.01',
        ]);

        $task->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Görev başarıyla güncellendi.',
            'task'    => $task
        ]);
    }

    // Admin için görev detayı
    public function adminShow(WatchTask $task)
    {
        $completions = $task->completions()
                          ->with('user:id,account_holder')
                          ->where('is_completed', true)
                          ->latest()
                          ->limit(10)
                          ->get();

        return response()->json([
            'success' => true,
            'task' => [
                'id'                  => $task->id,
                'title'               => $task->title,
                'description'         => $task->description,
                'youtube_url'         => $task->youtube_url,
                'youtube_video_id'    => $task->youtube_video_id,
                'watch_duration'      => $task->watch_duration,
                'formatted_duration'  => $task->formatted_duration,
                'reward_amount'       => $task->reward_amount,
                'max_completions'     => $task->max_completions,
                'current_completions' => $task->current_completions,
                'is_active'           => $task->is_active,
                'created_at'          => $task->created_at->format('d.m.Y H:i'),
                'creator'             => $task->creator->account_holder,
            ],
            'completions' => $completions
        ]);
    }
}
