<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\City;
use App\Models\Country;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use App\Models\UserSession;
use Illuminate\Support\Facades\Log;
use App\Http\Helpers\RecaptchaHelper;

class LoginController extends Controller
{
 
    public function showLoginForm()
    {
        
        $countries = Country::all(); 
        $cities = City::all(); 
        return view('auth.login', compact('countries','cities'));
    }

   
    public function login(Request $request)
    {
        try {
            $credentials = $request->validate([
                'email' => 'required|email',
                'password' => 'required',
            ]);
            
            // reCAPTCHA doğrulaması
            if (!RecaptchaHelper::verify($request->input('g-recaptcha-response'))) {
                return response()->json([
                    'success' => false,
                    'message' => __('reCAPTCHA verification failed. Please try again.')
                ]);
            }
    
            // Önce kullanıcıyı bul
            $user = \App\Models\User::where('email', $credentials['email'])->first();
            
            // Remember me kontrolü için log
            $rememberMe = $request->filled('remember');
            Log::info('Login attempt', [
                'email' => $credentials['email'],
                'remember_me' => $rememberMe,
                'user_found' => $user ? 'yes' : 'no'
            ]);
            
            if ($user) {
                // Customer kullanıcıların primary site'a girişini engelle
                if ($this->isCustomerUser($user)) {
                    return response()->json([
                        'success' => false,
                        'message' => __('Customer accounts cannot access this area. Please use the customer portal.')
                    ]);
                }
                
                // Eğer password_updated = 0 ise MD5 şifre kontrolü yap
                if ($user->password_updated == 0 && !empty($user->sifre)) {
                    $md5Password = md5($credentials['password']);
                    
                    if ($md5Password === $user->sifre) {
                        // MD5 şifre eşleşti, yeni şifreyi güncelle
                        $user->password = \Hash::make($credentials['password']);
                        $user->password_updated = 1;
                        $user->sifre = null; // MD5 şifreyi temizle
                        $user->save();
                        
                        // Kullanıcıyı giriş yaptır
                        $rememberMe = $request->filled('remember');
                        Log::info('MD5 login attempt', ['user_id' => $user->id, 'remember_me' => $rememberMe]);
                        
                        Auth::login($user, $rememberMe);
                        $request->session()->regenerate();
                        UserSession::updateOrCreate(
                            ['user_id' => auth()->id()],
                            ['last_active' => now(), 'is_active' => true]
                        );
                        
                        // Günlük bonus kontrolü
                        try {
                            $bonusResult = $user->checkAndAddDailyBonus();
                            $message = __('Password updated successfully!');
                            if ($bonusResult) {
                                // Bonus bilgisini session'a kaydet
                                $request->session()->flash('daily_bonus', [
                                    'amount' => $bonusResult['amount'],
                                    'currency' => \App\Models\Setting::first()->pbirim ?? 'TL'
                                ]);
                            }
                        } catch (\Exception $bonusError) {
                            Log::error('Daily bonus error for user ' . $user->id . ': ' . $bonusError->getMessage());
                            $bonusResult = false;
                            $message = __('Password updated successfully!');
                        }
                        
                        return response()->json([
                            'success' => true, 
                            'redirect' => '/dashboard',
                            'message' => $message
                        ]);
                    }
                }
                
                // Normal Laravel Auth kontrolü (password_updated = 1 olanlar için)
                $rememberMe = $request->filled('remember');
                Log::info('Normal login attempt', ['email' => $credentials['email'], 'remember_me' => $rememberMe]);
                
                if (Auth::attempt($credentials, $rememberMe)) {
                    // Giriş başarılı olduktan sonra da customer kontrolü yap
                    $authenticatedUser = Auth::user();
                    if ($this->isCustomerUser($authenticatedUser)) {
                        Auth::logout();
                        return response()->json([
                            'success' => false,
                            'message' => __('Customer accounts cannot access this area. Please use the customer portal.')
                        ]);
                    }
                    
                    $request->session()->regenerate();
                    UserSession::updateOrCreate(
                        ['user_id' => auth()->id()],
                        ['last_active' => now(), 'is_active' => true]
                    );
                    
                    // Günlük bonus kontrolü
                    $user = Auth::user();
                    try {
                        $bonusResult = $user->checkAndAddDailyBonus();
                        $message = __('Login successful!');
                        if ($bonusResult) {
                            // Bonus bilgisini session'a kaydet
                            $request->session()->flash('daily_bonus', [
                                'amount' => $bonusResult['amount'],
                                'currency' => \App\Models\Setting::first()->pbirim ?? 'TL'
                            ]);
                        }
                    } catch (\Exception $bonusError) {
                        Log::error('Daily bonus error for user ' . $user->id . ': ' . $bonusError->getMessage());
                        $bonusResult = false;
                        $message = __('Login successful!');
                    }
                    
                    return response()->json([
                        'success' => true, 
                        'redirect' => '/dashboard',
                        'message' => $message
                    ]);
                }
            }
    
            return response()->json([
                'success' => false,
                'message' => __('The provided credentials do not match our records.')
            ]);
    
        } catch (\Exception $e) {
            Log::error('Login error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'email' => $request->email ?? 'N/A',
                'user_agent' => $request->userAgent(),
                'ip' => $request->ip()
            ]);
            return response()->json([
                'success' => false, 
                'message' => __('An error occurred during login. Please try again.'),
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
  
    public function logout(Request $request)
{
    UserSession::where('user_id', auth()->id())->update(['is_active' => false]);
    Auth::logout();
    $request->session()->flush();
    $request->session()->invalidate();
    $request->session()->regenerateToken();
    return redirect('/');
}

    /**
     * Customer kullanıcı kontrolü (membership_type bazlı)
     */
    private function isCustomerUser($user)
    {
        if (!$user) {
            return false;
        }

        // membership_type kontrolü (customer veya 3)
        if (isset($user->membership_type)) {
            $membershipType = (string)$user->membership_type;
            if (in_array($membershipType, ['3', 'customer'], true)) {
                return true;
            }
        }

        return false;
    }
}