<?php

namespace App\Http\Controllers\GorevSitesi;

use App\Http\Controllers\Controller;
use App\Models\Country;
use App\Models\City;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class AuthController extends Controller
{
    public function showLoginForm()
    {
        return view('gorevsitesi.auth.login');
    }

    
    public function showRegistrationForm()
    {
        $countries = Country::orderBy('name')->get(['id','name','code']);
        return view('gorevsitesi.auth.register', compact('countries'));
    }

    public function getCitiesByCountryCode(string $countryCode)
    {
        $cities = City::select('id','name')
            ->where('country_id', $countryCode)
            ->orderBy('name')
            ->get();

        return response()->json(['cities' => $cities]);
    }

    public function register(Request $request)
    {
        $request->validate([
            'account_holder' => ['required','string','max:190'],
            'username'       => ['required','string','max:190','unique:users,username'],
            'email'          => ['required','email','max:190','unique:users,email'],
            'phone'          => ['required','string','max:190','unique:users,phone'],
            'country_code'   => ['required','string','max:10'],
            'city_id'        => ['required','integer'],
            'gender'         => ['required','in:male,female,other'],
            'password'       => ['required','confirmed','min:8'],
        ]);

        $user = User::create([
            'account_holder'  => $request->input('account_holder'),
            'username'        => $request->input('username'),
            'email'           => $request->input('email'),
            'phone'           => $request->input('phone'),
            'country_id'      => $request->input('country_code'), // City.country_id -> Countries.code
            'city_id'         => $request->input('city_id'),
            'gender'          => $request->input('gender'),
            'membership_type' => '3', // müşteri
            'password'        => Hash::make($request->input('password')),
        ]);

        return redirect()
            ->route('site2.login')
            ->with('status', __('Your account has been created. You can login now.'));
    }
public function login(Request $request)
{

    $request->validate([
        'username' => ['required','string'],
        'password' => ['required','string'],
    ]);

    // email mi yoksa username mi?
    $field = filter_var($request->username, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';

    $credentials = [
        $field    => $request->username,
        'password'=> $request->password,
    ];

    $ok = Auth::attempt($credentials, $request->boolean('remember'));
    Log::info('SC login attempt', ['ok' => $ok, 'user' => $ok ? Auth::id() : null]);

    if (!$ok) {
        return back()
            ->withErrors(['username' => __('These credentials do not match our records.')])
            ->onlyInput('username');
    }

    // Oturum sabitleme saldırılarına karşı
    $request->session()->regenerate();

  $type = (string) (Auth::user()->membership_type ?? '');
if (!in_array($type, ['3', 'customer'], true)) {
    Auth::logout();
    $request->session()->invalidate();
    $request->session()->regenerateToken();

    return redirect()
        ->route('site2.login')
        ->withErrors(['auth' => __('Your account is not allowed to access customer area.')]);
}

return redirect()->route('site2.account.dashboard');
}

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('site2.landing');
    }
}
