<?php

namespace App\Http\Controllers\GorevSitesi;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\BalanceLog;
use App\Models\TransactionLog;

class BalanceController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'is_customer']);
    }

    public function index()
    {
        $user = Auth::user();
        $transactions = BalanceLog::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('gorevsitesi.account.balance.index', compact('user', 'transactions'));
    }

    public function create()
    {
        $user = Auth::user();
        return view('gorevsitesi.account.balance.create', compact('user'));
    }

    public function store(Request $request)
    {
        $validationRules = [
            'amount' => 'required|numeric|min:10|max:10000',
            'payment_method' => 'required|in:credit_card,bank_transfer'
        ];

        // Banka havalesi için dekont zorunlu
        if ($request->payment_method === 'bank_transfer') {
            $validationRules['receipt'] = 'required|file|mimes:jpg,jpeg,png,pdf|max:5120'; // 5MB max
        }

        // Kredi kartı için kart bilgileri zorunlu
        if ($request->payment_method === 'credit_card') {
            $validationRules = array_merge($validationRules, [
                'card_number' => 'required|string|size:19', // 1234 5678 9012 3456 format
                'card_holder' => 'required|string|max:255',
                'expiry_month' => 'required|string|size:2',
                'expiry_year' => 'required|string|size:2',
                'cvv' => 'required|string|size:3'
            ]);
        }

        // AJAX request için validation
        if ($request->ajax()) {
            $validator = \Validator::make($request->all(), $validationRules);
            
            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
        } else {
            $request->validate($validationRules);
        }

        $user = Auth::user();
        $amount = $request->amount;

        try {
            if ($request->payment_method === 'bank_transfer') {
                // Banka havalesi - Onay bekleyen durum
                $receiptPath = null;
                if ($request->hasFile('receipt')) {
                    $receiptPath = $request->file('receipt')->store('balance_receipts', 'public');
                }

                // Onay bekleyen balance log
                BalanceLog::create([
                    'user_id' => $user->id,
                    'amount' => $amount,
                    'type' => 'deposit',
                    'description' => 'Banka havalesi ile bakiye yükleme',
                    'status' => 'pending',
                    'receipt_path' => $receiptPath,
                    'payment_method' => 'bank_transfer'
                ]);

                if ($request->ajax()) {
                    return response()->json([
                        'success' => true,
                        'message' => 'Banka havalesi talebiniz alındı. Dekontunuz incelendikten sonra bakiyeniz yüklenecektir.',
                        'redirect' => route('site2.account.dashboard')
                    ]);
                }
                
                return redirect()->route('site2.account.dashboard')
                    ->with('info', 'Banka havalesi talebiniz alındı. Dekontunuz incelendikten sonra bakiyeniz yüklenecektir.');

            } else {
                // Kredi kartı - Anında işlem (simülasyon)
                // Gerçek uygulamada ödeme gateway'i entegrasyonu yapılacak
                
                // Kart bilgilerini logla (güvenlik için hash'le)
                $cardInfo = [
                    'card_last4' => substr(str_replace(' ', '', $request->card_number), -4),
                    'card_holder' => $request->card_holder,
                    'expiry' => $request->expiry_month . '/' . $request->expiry_year
                ];

                // Başarılı ödeme simülasyonu
                BalanceLog::create([
                    'user_id' => $user->id,
                    'amount' => $amount,
                    'type' => 'deposit',
                    'description' => 'Kredi kartı ile bakiye yükleme (*' . $cardInfo['card_last4'] . ')',
                    'status' => 'completed',
                    'payment_method' => 'credit_card',
                    'card_info' => json_encode($cardInfo)
                ]);

                // Bakiyeyi güncelle
                $user->increment('balance', $amount);

                // Transaction log
                TransactionLog::create([
                    'user_id' => $user->id,
                    'type' => 'balance_deposit',
                    'amount' => $amount,
                    'description' => 'Kredi kartı ile bakiye yükleme',
                    'status' => 'completed'
                ]);

                if ($request->ajax()) {
                    return response()->json([
                        'success' => true,
                        'message' => $amount . ' ₺ bakiye başarıyla yüklendi.',
                        'redirect' => route('site2.account.dashboard')
                    ]);
                }
                
                return redirect()->route('site2.account.dashboard')
                    ->with('success', $amount . ' ₺ bakiye başarıyla yüklendi.');
            }

        } catch (\Exception $e) {
            \Log::error('Balance deposit error: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'amount' => $amount,
                'payment_method' => $request->payment_method
            ]);
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bakiye yükleme işlemi başarısız. Lütfen tekrar deneyin.'
                ], 500);
            }
            
            return back()->with('error', 'Bakiye yükleme işlemi başarısız. Lütfen tekrar deneyin.');
        }
    }
}
