<?php

namespace App\Http\Controllers;

use App\Models\UserSession;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Models\SocialPlatform;
use Illuminate\Support\Facades\Cache;
use App\Models\User;
use App\Models\Task;
use App\Models\TaskProof;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Helpers\UserHelper;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class HomeController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth')->except('landing');
    }

    public function landing()
    {
        // Eğer kullanıcı giriş yapmışsa dashboard'a yönlendir
        if (Auth::check()) {
            return redirect()->route('task.index');
        }

        // Landing page için istatistikler
        $stats = [
            'total_users' => \App\Models\User::count(),
            'verified_users' => \App\Models\User::where('phone_verified', true)->count(),
            'total_tasks' => \App\Models\Task::count() + \App\Models\WebsiteVisitTask::count(),
            'completed_tasks' => \App\Models\TaskUser::where('is_completed', true)->count() + 
                               \App\Models\WebsiteVisit::where('status', 'completed')->count(),
            'website_visits' => \App\Models\WebsiteVisit::where('status', 'completed')->count(),
            'social_tasks' => \App\Models\TaskUser::where('is_completed', true)->count(),
            'total_rewards' => \App\Models\BalanceLog::sum('balance') - \App\Models\BalanceLog::sum('balance_before')
        ];

        return view('landing', compact('stats'));
    }

    public function index()
    {
        $currentUser = UserHelper::getCurrentUser(); // Helper kullanarak kullanıcı bilgisini al
        $startDate = Carbon::now()->startOfWeek();
        $endDate = Carbon::now()->endOfWeek();
        $platforms = SocialPlatform::where('status', 1)->get();
        $mobileverifed = User::where('phone_verified', 1)->count();
        $emailverifed = User::where('verified', 1)->count();
        $totaluser = User::where('membership_type', 1)->count();
        $totalBalance = User::sum('balance');
        $todayRegistrations = User::whereDate('created_at', today())->count();
        $tasksAwaitingApproval = Task::where('status', 0)->count();
        $activeTasks = Task::where('status', 1)->count();
        $completedTasks = Task::where('status', 2)->count();
        $weeklyTask = Task::whereBetween('created_at', [$startDate, $endDate])->count();
        $onlineUsersCount = UserSession::where('is_active', true)->count();
        $evidencesAwaitingApproval = TaskProof::where('status', 0)->count();
        // Son eklenen görevleri çek
        $recentTasks = $this->getRecentTasks();

        return view('dashboard', compact('platforms', 'currentUser', 'onlineUsersCount', 'mobileverifed', 'emailverifed', 'totaluser', 'totalBalance', 'todayRegistrations', 'tasksAwaitingApproval', 'activeTasks', 'completedTasks', 'weeklyTask','evidencesAwaitingApproval', 'recentTasks'));
    }

    private function getOnlineUserCount()
    {
        $onlineUsers = Cache::get('online-users', []);
        return count($onlineUsers);
    }

    /**
     * API endpoint for DataTables - Users list
     */
    public function getUsersDataTable(Request $request)
    {
        $query = User::select(
            'users.id',
            'users.account_holder',
            'users.username', 
            'users.email',
            'users.phone',
            'users.phone_verified',
            'users.gender',
            'users.balance',
            'users.created_at',
            'cities.name as city_name', 
            'countries.name as country_name'
        )
        ->leftJoin('cities', 'cities.id', '=', 'users.city_id')
        ->leftJoin('countries', 'countries.code', '=', 'users.country_id')
        ->where('users.user_type', '!=', 1); // Admin olmayan kullanıcılar

        // Search functionality
        if ($request->has('search') && !empty($request->search['value'])) {
            $searchValue = $request->search['value'];
            $query->where(function($q) use ($searchValue) {
                $q->where('users.account_holder', 'like', "%{$searchValue}%")
                  ->orWhere('users.username', 'like', "%{$searchValue}%")
                  ->orWhere('users.email', 'like', "%{$searchValue}%")
                  ->orWhere('users.phone', 'like', "%{$searchValue}%")
                  ->orWhere('cities.name', 'like', "%{$searchValue}%")
                  ->orWhere('countries.name', 'like', "%{$searchValue}%");
            });
        }

        // Ordering
        if ($request->has('order')) {
            $orderColumn = $request->order[0]['column'];
            $orderDir = $request->order[0]['dir'];
            
            $columns = ['id', 'account_holder', 'username', 'email', 'phone', 'phone_verified', 'gender', 'balance', 'created_at'];
            if (isset($columns[$orderColumn])) {
                $query->orderBy('users.' . $columns[$orderColumn], $orderDir);
            }
        } else {
            $query->orderBy('users.created_at', 'desc');
        }

        // Get total count before pagination
        $totalRecords = User::where('user_type', '!=', 1)->count();
        $filteredRecords = $query->count();

        // Pagination
        $start = $request->start ?? 0;
        $length = $request->length ?? 10;
        $users = $query->skip($start)->take($length)->get();

        // Format data for DataTables
        $data = [];
        foreach ($users as $user) {
            $data[] = [
                'account_holder' => $user->account_holder,
                'username' => '#' . $user->username,
                'email' => $user->email,
                'location' => ($user->country_name ?? 'N/A') . '/' . ($user->city_name ?? 'N/A'),
                'phone' => '<a href="https://wa.me/' . $user->phone . '" target="_blank" title="WhatsApp"><i class="lni lni-whatsapp"></i> ' . $user->phone . '</a>',
                'phone_verified' => '<div class="' . ($user->phone_verified ? 'btn btn-success' : 'btn btn-danger') . '">' . 
                                   ($user->phone_verified ? __('Verified') : __('Not Verified')) . '</div>',
                'gender' => '<div class="' . ($user->gender == 'male' ? 'badge bg-gradient-quepal text-white shadow-sm w-100' : 'badge bg-gradient-bloody text-white shadow-sm w-100') . '">' .
                           ($user->gender == 'male' ? __('Male') : __('Female')) . '</div>',
                'balance' => number_format($user->balance, 2),
                'created_at' => $user->created_at->format('d.m.Y H:i')
            ];
        }

        return response()->json([
            'draw' => intval($request->draw),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data
        ]);
    }

    /**
     * Son eklenen görevleri getir (Sosyal medya, İzle Kazan, Web sitesi ziyaret)
     */
    private function getRecentTasks()
    {
        $recentTasks = collect();

        // Sosyal medya görevleri (Task tablosu)
        $socialTasks = \App\Models\Task::select(
            'id',
            'name as title',
            'description',
            'reward',
            'created_at',
            'status'
        )
        ->selectRaw("'social' as task_type")
        ->where('status', 1)
        ->orderBy('created_at', 'desc')
        ->limit(3)
        ->get();

        // Web sitesi ziyaret görevleri
        $websiteTasks = \App\Models\WebsiteVisitTask::select(
            'id',
            'title',
            'description',
            'reward_amount as reward',
            'created_at',
            'status'
        )
        ->selectRaw("'website' as task_type")
        ->where('status', 1)
        ->orderBy('created_at', 'desc')
        ->limit(2)
        ->get();

        // Görevleri birleştir
        $recentTasks = $socialTasks->concat($websiteTasks);

        // Tarihe göre sırala ve ilk 5'ini al
        return $recentTasks->sortByDesc('created_at')->take(5)->values();
    }

    /**
     * API endpoint for refreshing recent tasks
     */
    public function getRecentTasksApi()
    {
        $recentTasks = $this->getRecentTasks();
        $settings = app('settings');
        
        $html = '';
        
        if ($recentTasks->count() > 0) {
            foreach ($recentTasks as $task) {
                $taskTypeLabel = '';
                $taskId = '';
                $viewUrl = '';
                
                if ($task->task_type == 'social') {
                    $taskTypeLabel = '<span class="badge bg-gradient-quepal text-white">' . __('Social Media') . '</span>';
                    $taskId = '#SM' . $task->id;
                    $viewUrl = route('tasks.show', $task->id);
                } elseif ($task->task_type == 'website') {
                    $taskTypeLabel = '<span class="badge bg-gradient-blooker text-white">' . __('Website Visit') . '</span>';
                    $taskId = '#WV' . $task->id;
                    $viewUrl = route('user.website-visits.show', $task->id);
                } else {
                    $taskTypeLabel = '<span class="badge bg-gradient-bloody text-white">' . __('Watch & Earn') . '</span>';
                    $taskId = '#WE' . $task->id;
                    $viewUrl = '#';
                }
                
                $html .= '<tr>';
                $html .= '<td>';
                $html .= '<div class="d-flex align-items-center">';
                $html .= '<div class="ms-2">';
                $html .= '<h6 class="mb-0 font-14">' . Str::limit($task->title, 30) . '</h6>';
                if ($task->description) {
                    $html .= '<p class="mb-0 font-13 text-secondary">' . Str::limit($task->description, 50) . '</p>';
                }
                $html .= '</div>';
                $html .= '</div>';
                $html .= '</td>';
                $html .= '<td>' . $taskTypeLabel . '</td>';
                $html .= '<td>' . $taskId . '</td>';
                $html .= '<td><span class="badge bg-gradient-quepal text-white shadow-sm w-100">' . __('Active') . '</span></td>';
                $html .= '<td>' . number_format($task->reward, 2, ',', '.') . ($settings['pbirim'] ?? 'TL') . '</td>';
                $html .= '<td>' . $this->formatDateByLocale($task->created_at) . '</td>';
                $html .= '<td>';
                if ($task->task_type == 'social' || $task->task_type == 'website') {
                    $html .= '<a href="' . $viewUrl . '" class="btn btn-primary btn-sm">' . __('View') . '</a>';
                }
                $html .= '</td>';
                $html .= '</tr>';
            }
        } else {
            $html = '<tr>';
            $html .= '<td colspan="7" class="text-center py-4">';
            $html .= '<i class="bx bx-info-circle me-2"></i>' . __('No recent tasks found');
            $html .= '</td>';
            $html .= '</tr>';
        }
        
        return response()->json([
            'success' => true,
            'html' => $html
        ]);
    }

    /**
     * Format date according to current locale
     */
    private function formatDateByLocale($date)
    {
        $locale = app()->getLocale();
        
        // Ay isimlerini dillere göre tanımla
        $months = [
            'tr' => [
                1 => 'Oca', 2 => 'Şub', 3 => 'Mar', 4 => 'Nis', 5 => 'May', 6 => 'Haz',
                7 => 'Tem', 8 => 'Ağu', 9 => 'Eyl', 10 => 'Eki', 11 => 'Kas', 12 => 'Ara'
            ],
            'en' => [
                1 => 'Jan', 2 => 'Feb', 3 => 'Mar', 4 => 'Apr', 5 => 'May', 6 => 'Jun',
                7 => 'Jul', 8 => 'Aug', 9 => 'Sep', 10 => 'Oct', 11 => 'Nov', 12 => 'Dec'
            ],
            'ru' => [
                1 => 'Янв', 2 => 'Фев', 3 => 'Мар', 4 => 'Апр', 5 => 'Май', 6 => 'Июн',
                7 => 'Июл', 8 => 'Авг', 9 => 'Сен', 10 => 'Окт', 11 => 'Ноя', 12 => 'Дек'
            ],
            'ar' => [
                1 => 'يناير', 2 => 'فبراير', 3 => 'مارس', 4 => 'أبريل', 5 => 'مايو', 6 => 'يونيو',
                7 => 'يوليو', 8 => 'أغسطس', 9 => 'سبتمبر', 10 => 'أكتوبر', 11 => 'نوفمبر', 12 => 'ديسمبر'
            ]
        ];
        
        $day = $date->format('d');
        $month = $months[$locale][$date->format('n')] ?? $months['en'][$date->format('n')];
        $year = $date->format('Y');
        
        // Arapça için farklı format (sağdan sola)
        if ($locale === 'ar') {
            return $year . ' ' . $month . ' ' . $day;
        }
        
        return $day . ' ' . $month . ' ' . $year;
    }

    /**
     * AJAX: Son üyeleri getir (sadece admin)
     */
    public function getRecentMembersAjax(Request $request)
    {
        $user = auth()->user();
        Log::info('Recent members request from user: ' . $user->id . ', type: ' . $user->user_type . ', isAdmin: ' . ($user->user_type == 1 ? 'true' : 'false'));
        
        // Admin kontrolü - user_type 1 olmalı
        if (!$user || $user->user_type != 1) {
            Log::warning('Unauthorized access to recent members by user: ' . ($user ? $user->id : 'null') . ', user_type: ' . ($user ? $user->user_type : 'null'));
            return response()->json(['error' => 'Unauthorized - Admin access required'], 403);
        }

        try {
            $members = User::select(
                'id', 'account_holder', 'username', 'email', 'phone', 
                'phone_verified', 'gender', 'balance', 'created_at',
                'city_id', 'country_id'
            )
            ->with(['city:id,name', 'country:code,name'])
            ->where('user_type', '!=', 1)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
            
            Log::info('Found ' . $members->count() . ' recent members');
            
            $membersData = $members->map(function($user) {
                return [
                    'id' => $user->id,
                    'name' => $user->account_holder,
                    'username' => $user->username,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'phone_verified' => $user->phone_verified,
                    'gender' => $user->gender,
                    'balance' => number_format($user->balance, 2),
                    'location' => ($user->country->name ?? '') . ($user->city ? ' / ' . $user->city->name : ''),
                    'join_date' => $user->created_at->format('d.m.Y'),
                    'join_date_human' => $user->created_at->diffForHumans(),
                    'avatar' => 'https://ui-avatars.com/api/?name=' . urlencode($user->account_holder) . '&background=random'
                ];
            });

            Log::info('Returning ' . $membersData->count() . ' members data');
            return response()->json($membersData);
            
        } catch (\Exception $e) {
            Log::error('Error fetching recent members: ' . $e->getMessage());
            return response()->json(['error' => 'Server error'], 500);
        }
    }

    /**
     * AJAX: Son görevleri getir (mobile-friendly)
     */
    public function getRecentTasksAjax(Request $request)
    {
        $tasks = $this->getRecentTasks()->map(function($task) {
            return [
                'id' => $task->id,
                'title' => $task->title,
                'description' => Str::limit($task->description ?? '', 50),
                'type' => $task->task_type,
                'type_text' => $task->task_type == 'social' ? __('Social Media') : __('Website Visit'),
                'type_badge' => $task->task_type == 'social' ? 'bg-gradient-quepal' : 'bg-gradient-blooker',
                'task_id' => '#' . ($task->task_type == 'social' ? 'SM' : 'WV') . $task->id,
                'reward' => number_format($task->reward, 2),
                'currency' => app('settings')['pbirim'] ?? 'TL',
                'date' => $task->created_at->format('d.m.Y'),
                'date_human' => $task->created_at->diffForHumans(),
                'view_url' => $task->task_type == 'social' 
                    ? route('tasks.show', $task->id)
                    : route('user.website-visits.show', $task->id)
            ];
        });

        return response()->json($tasks);
    }
}