<?php

namespace App\Http\Controllers\Marketplace;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Order;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class MarketplaceController extends Controller
{
    /**
     * Marketplace landing page
     */
    public function index()
    {
        $featuredProducts = Product::where('status', 1)
            ->where('is_featured', true)
            ->take(8)
            ->get();
            
        $categories = \App\Models\ProductCategory::where('status', 1)
            ->withCount('products')
            ->take(6)
            ->get();
            
        $stats = [
            'total_products' => Product::where('status', 1)->count(),
            'total_customers' => User::where('user_type', 3)->count(),
            'total_orders' => Order::count(),
            'happy_customers' => User::where('user_type', 3)->where('created_at', '>=', now()->subMonth())->count()
        ];

        return view('marketplace.landing', compact('featuredProducts', 'categories', 'stats'));
    }

    /**
     * Products listing page
     */
    public function products(Request $request)
    {
        $query = Product::where('status', 1);
        
        // Category filter
        if ($request->category) {
            $query->where('category_id', $request->category);
        }
        
        // Search filter
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('product_name', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
            });
        }
        
        // Price filter
        if ($request->min_price) {
            $query->where('price', '>=', $request->min_price);
        }
        if ($request->max_price) {
            $query->where('price', '<=', $request->max_price);
        }
        
        // Sorting
        switch ($request->sort) {
            case 'price_low':
                $query->orderBy('price', 'asc');
                break;
            case 'price_high':
                $query->orderBy('price', 'desc');
                break;
            case 'newest':
                $query->orderBy('created_at', 'desc');
                break;
            case 'popular':
                $query->withCount('orders')->orderBy('orders_count', 'desc');
                break;
            default:
                $query->orderBy('created_at', 'desc');
        }
        
        $products = $query->paginate(12);
        $categories = \App\Models\ProductCategory::where('status', 1)->get();
        
        return view('marketplace.products.index', compact('products', 'categories'));
    }

    /**
     * Product detail page
     */
    public function productDetail($id)
    {
        $product = Product::where('status', 1)->findOrFail($id);
        
        // Related products
        $relatedProducts = Product::where('status', 1)
            ->where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->take(4)
            ->get();
            
        return view('marketplace.products.detail', compact('product', 'relatedProducts'));
    }

    /**
     * User registration for marketplace
     */
    public function register(Request $request)
    {
        if ($request->isMethod('post')) {
            $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:users',
                'password' => 'required|string|min:8|confirmed',
                'phone' => 'required|string|max:20',
                'terms' => 'accepted'
            ]);

            $user = User::create([
                'account_holder' => $request->name,
                'username' => $this->generateUsername($request->name),
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'phone' => $request->phone,
                'user_type' => 3, // Marketplace customer
                'status' => 1,
                'email_verified_at' => now()
            ]);

            Auth::login($user);

            return redirect()->route('marketplace.dashboard')
                ->with('success', __('Registration successful! Welcome to TaskZone Marketplace.'));
        }

        return view('marketplace.auth.register');
    }

    /**
     * User login for marketplace
     */
    public function login(Request $request)
    {
        if ($request->isMethod('post')) {
            $request->validate([
                'email' => 'required|email',
                'password' => 'required',
            ]);

            $credentials = $request->only('email', 'password');
            $credentials['user_type'] = 3; // Only marketplace users

            if (Auth::attempt($credentials, $request->remember)) {
                $request->session()->regenerate();
                return redirect()->intended(route('marketplace.dashboard'));
            }

            return back()->withErrors([
                'email' => __('The provided credentials do not match our records.'),
            ]);
        }

        return view('marketplace.auth.login');
    }

    /**
     * User dashboard
     */
    public function dashboard()
    {
        $user = Auth::user();
        
        $recentOrders = Order::where('user_id', $user->id)
            ->with('product')
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();
            
        $stats = [
            'total_orders' => Order::where('user_id', $user->id)->count(),
            'pending_orders' => Order::where('user_id', $user->id)->where('status', 'pending')->count(),
            'completed_orders' => Order::where('user_id', $user->id)->where('status', 'completed')->count(),
            'total_spent' => Order::where('user_id', $user->id)->where('status', 'completed')->sum('total_amount')
        ];

        return view('marketplace.dashboard', compact('recentOrders', 'stats'));
    }

    /**
     * User orders
     */
    public function orders()
    {
        $orders = Order::where('user_id', Auth::id())
            ->with('product')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('marketplace.orders.index', compact('orders'));
    }

    /**
     * Order detail
     */
    public function orderDetail($id)
    {
        $order = Order::where('user_id', Auth::id())
            ->with('product')
            ->findOrFail($id);

        return view('marketplace.orders.detail', compact('order'));
    }

    /**
     * Cancel order
     */
    public function cancelOrder($id)
    {
        $order = Order::where('user_id', Auth::id())
            ->where('status', 'pending')
            ->findOrFail($id);

        $order->update(['status' => 'cancelled']);

        return redirect()->route('marketplace.orders')
            ->with('success', __('Order cancelled successfully.'));
    }

    /**
     * Generate unique username
     */
    private function generateUsername($name)
    {
        $username = strtolower(str_replace(' ', '', $name));
        $originalUsername = $username;
        $counter = 1;

        while (User::where('username', $username)->exists()) {
            $username = $originalUsername . $counter;
            $counter++;
        }

        return $username;
    }
}