<?php

namespace App\Http\Controllers\Marketplace;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    /**
     * Purchase a product
     */
    public function purchase(Request $request, $productId)
    {
        $request->validate([
            'quantity' => 'required|integer|min:1|max:10'
        ]);

        $product = Product::where('status', 1)->findOrFail($productId);
        $quantity = $request->quantity;

        // Check stock
        if ($product->stock_quantity < $quantity) {
            return back()->with('error', __('Insufficient stock available.'));
        }

        DB::beginTransaction();
        try {
            // Calculate total
            $totalAmount = $product->price * $quantity;

            // Create order
            $order = Order::create([
                'order_number' => $this->generateOrderNumber(),
                'user_id' => Auth::id(),
                'product_id' => $product->id,
                'product_name' => $product->product_name,
                'product_price' => $product->price,
                'quantity' => $quantity,
                'total_amount' => $totalAmount,
                'status' => 'pending',
                'payment_status' => 'pending',
                'product_data' => json_encode([
                    'name' => $product->product_name,
                    'description' => $product->description,
                    'image' => $product->image,
                    'category' => $product->category->name ?? null
                ])
            ]);

            // Reduce stock
            $product->decrement('stock_quantity', $quantity);

            DB::commit();

            return redirect()->route('marketplace.checkout', $order)
                ->with('success', __('Product added to cart. Please complete your payment.'));

        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', __('Error processing your order. Please try again.'));
        }
    }

    /**
     * Checkout page
     */
    public function checkout(Order $order)
    {
        // Ensure order belongs to current user
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        // Ensure order is still pending
        if ($order->status !== 'pending') {
            return redirect()->route('marketplace.order.detail', $order)
                ->with('info', __('This order has already been processed.'));
        }

        return view('marketplace.checkout', compact('order'));
    }

    /**
     * Process payment
     */
    public function processPayment(Request $request, Order $order)
    {
        $request->validate([
            'payment_method' => 'required|in:balance,credit_card,bank_transfer'
        ]);

        // Ensure order belongs to current user
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        // Ensure order is still pending
        if ($order->status !== 'pending') {
            return redirect()->route('marketplace.order.detail', $order)
                ->with('info', __('This order has already been processed.'));
        }

        DB::beginTransaction();
        try {
            $paymentMethod = $request->payment_method;

            if ($paymentMethod === 'balance') {
                // Pay with user balance
                $user = Auth::user();
                
                if ($user->balance < $order->total_amount) {
                    return back()->with('error', __('Insufficient balance. Please choose another payment method.'));
                }

                // Deduct from balance
                $user->decrement('balance', $order->total_amount);

                // Log balance transaction
                \App\Models\BalanceLog::create([
                    'userId' => $user->id,
                    'amount' => -$order->total_amount,
                    'operation_type' => 'marketplace_purchase',
                    'description' => "Marketplace purchase: {$order->product_name}",
                    'reference_id' => $order->id,
                    'reference_type' => 'marketplace_order',
                    'balance_before' => $user->balance + $order->total_amount,
                    'balance' => $user->balance,
                    'added_user' => $user->id
                ]);

                // Update order
                $order->update([
                    'status' => 'completed',
                    'payment_status' => 'paid',
                    'payment_method' => 'balance',
                    'delivered_at' => now()
                ]);

                $message = __('Payment successful! Your order has been completed.');
                
            } else {
                // For other payment methods, mark as processing
                $order->update([
                    'status' => 'processing',
                    'payment_status' => 'pending',
                    'payment_method' => $paymentMethod
                ]);

                $message = __('Order submitted! We will process your payment and notify you once completed.');
            }

            DB::commit();

            return redirect()->route('marketplace.order.detail', $order)
                ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', __('Payment processing failed. Please try again.'));
        }
    }

    /**
     * Generate unique order number
     */
    private function generateOrderNumber()
    {
        do {
            $orderNumber = 'MP' . date('Ymd') . rand(1000, 9999);
        } while (Order::where('order_number', $orderNumber)->exists());

        return $orderNumber;
    }
}