<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Notification;
use App\Services\NotificationService;
use Illuminate\Support\Facades\Auth;

class NotificationController extends Controller
{
    protected $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Bildirimi okundu olarak işaretle
     */
    public function markAsRead($id)
    {
        $notification = Notification::where('id', $id)
            ->where('user_id', Auth::id())
            ->first();
            
        if ($notification) {
            $notification->update([
                'status' => 0, // 0 = read
                'read_at' => now()
            ]);
            return response()->json(['success' => true]);
        }
        return response()->json(['success' => false], 404);
    }

    /**
     * Tüm bildirimleri getir
     */
    public function index(Request $request)
    {
        $notifications = Notification::where('user_id', Auth::id())
            ->where(function($query) {
                $query->whereNull('expires_at')
                      ->orWhere('expires_at', '>', now());
            })
            ->orderBy('created_at', 'desc')
            ->paginate(20);
            
        $unreadCount = $this->notificationService->getUnreadCount(Auth::id());

        if ($request->expectsJson()) {
            return response()->json([
                'notifications' => $notifications,
                'unread_count' => $unreadCount
            ]);
        }

        return view('notifications.index', compact('notifications', 'unreadCount'));
    }

    /**
     * Tüm bildirimleri okundu olarak işaretle
     */
    public function markAllAsRead()
    {
        Notification::where('user_id', Auth::id())
            ->where('status', 1) // 1 = unread
            ->update([
                'status' => 0, // 0 = read
                'read_at' => now()
            ]);

        return response()->json(['success' => true]);
    }

    /**
     * Bildirimi sil
     */
    public function destroy($id)
    {
        $notification = Notification::where('id', $id)
            ->where('user_id', Auth::id())
            ->first();
            
        if ($notification) {
            $notification->delete();
            return response()->json(['success' => true]);
        }
        return response()->json(['success' => false], 404);
    }

    /**
     * Süresi dolmuş bildirimleri temizle (cron job için)
     */
    public function cleanExpired()
    {
        $deletedCount = $this->notificationService->cleanExpiredNotifications();
        return response()->json([
            'success' => true,
            'deleted_count' => $deletedCount
        ]);
    }
}
