<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    // User: Sipariş listesi
    public function index(Request $request)
    {
        $user = Auth::user();
        
        if ($user->user_type == 1) {
            // Admin view
            return $this->adminIndex($request);
        }
        
        // User view - Ajax için
        if ($request->ajax()) {
            return $this->getUserOrdersAjax($request);
        }
        
        return view('orders.index');
    }
    
    // User: Ajax sipariş listesi
    public function getUserOrdersAjax(Request $request)
    {
        $user = Auth::user();
        $page = $request->get('page', 1);
        $perPage = 10;
        
        $query = Order::where('user_id', $user->id)
            ->with(['product'])
            ->orderBy('created_at', 'desc');
            
        // Filters
        if ($request->status) {
            $query->where('status', $request->status);
        }
        
        $orders = $query->paginate($perPage, ['*'], 'page', $page);
        
        // Format orders data
        $formattedOrders = $orders->getCollection()->map(function($order) {
            return [
                'id' => $order->id,
                'order_number' => $order->order_number,
                'product_name' => $order->product_name,
                'quantity' => $order->quantity,
                'total_amount' => $order->total_amount,
                'formatted_total' => number_format($order->total_amount, 2) . ' ' . (app('settings')['currency'] ?? '₺'),
                'status' => $order->status,
                'status_text' => $order->status_text,
                'status_badge' => $order->status_badge,
                'payment_status' => $order->payment_status,
                'payment_status_text' => $order->payment_status_text,
                'payment_status_badge' => $order->payment_status_badge,
                'created_at' => $order->created_at->format('d.m.Y H:i'),
                'delivered_at' => $order->delivered_at ? $order->delivered_at->format('d.m.Y H:i') : null,
                'can_be_cancelled' => $order->canBeCancelled()
            ];
        });
        
        return response()->json([
            'success' => true,
            'orders' => $formattedOrders,
            'has_more' => $orders->hasMorePages(),
            'current_page' => $orders->currentPage()
        ]);
    }
    
    // Admin: Sipariş yönetimi
    public function adminIndex(Request $request)
    {
        // Statistics
        $stats = [
            'total' => Order::count(),
            'pending' => Order::where('status', 'pending')->count(),
            'processing' => Order::where('status', 'processing')->count(),
            'completed' => Order::where('status', 'completed')->count(),
            'revenue_today' => Order::whereDate('created_at', today())
                ->where('payment_status', 'paid')
                ->sum('total_amount'),
            'revenue_month' => Order::whereMonth('created_at', now()->month)
                ->where('payment_status', 'paid')
                ->sum('total_amount')
        ];
        
        return view('orders.admin.index', compact('stats'));
    }
    
    // Sipariş oluştur
    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1'
        ]);
        
        $user = Auth::user();
        $product = Product::findOrFail($request->product_id);
        
        // Check if product can be purchased
        if (!$product->canBePurchased($request->quantity)) {
            return response()->json([
                'success' => false,
                'message' => __('Product is not available for purchase')
            ], 400);
        }
        
        $totalAmount = $product->price * $request->quantity;
        
        // Check user balance
        if ($user->balance < $totalAmount) {
            return response()->json([
                'success' => false,
                'message' => __('Insufficient balance'),
                'required_amount' => $totalAmount,
                'current_balance' => $user->balance
            ], 400);
        }
        
        DB::beginTransaction();
        try {
            // Create order
            $order = Order::create([
                'order_number' => Order::generateOrderNumber(),
                'user_id' => $user->id,
                'product_id' => $product->id,
                'product_name' => $product->product_name,
                'product_price' => $product->price,
                'quantity' => $request->quantity,
                'total_amount' => $totalAmount,
                'status' => 'pending',
                'payment_status' => 'pending',
                'product_data' => [
                    'name' => $product->product_name,
                    'description' => $product->description,
                    'category' => $product->category->name ?? null,
                    'digital_content' => $product->digital_content,
                    'auto_delivery' => $product->auto_delivery
                ]
            ]);
            
            // Deduct balance
            $user->decrement('balance', $totalAmount);
            
            // Log balance transaction
            $balanceBefore = $user->balance + $totalAmount; // Çünkü zaten decrement yapıldı
            \App\Models\BalanceLog::create([
                'userId' => $user->id,
                'amount' => -$totalAmount,
                'operation_type' => 'purchase',
                'description' => "Product purchase: {$product->product_name}",
                'reference_id' => $order->id,
                'reference_type' => 'order',
                'balance_before' => $balanceBefore,
                'balance' => $user->balance,
                'added_user' => \Auth::id()
            ]);
            
            // Update order payment status
            $order->update(['payment_status' => 'paid']);
            
            // Decrease stock if not digital
            $product->decreaseStock($request->quantity);
            
            // Tüm siparişler processing durumunda başlar (admin onayı gerekli)
            $order->update(['status' => 'processing']);
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => __('Order created successfully'),
                'order' => [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'total_amount' => $order->formatted_total,
                    'status' => $order->status_text
                ]
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('Error creating order')
            ], 500);
        }
    }
    
    // Sipariş detayı
    public function show($id)
    {
        $user = Auth::user();
        
        if ($user->user_type == 1) {
            $order = Order::with(['user', 'product'])->findOrFail($id);
        } else {
            $order = Order::where('user_id', $user->id)
                ->with(['product'])
                ->findOrFail($id);
        }
        
        // AJAX request için JSON response
        if (request()->ajax() || request()->expectsJson()) {
            $orderData = [
                'id' => $order->id,
                'order_number' => $order->order_number,
                'product_name' => $order->product_name,
                'product_price' => $order->product_price,
                'quantity' => $order->quantity,
                'total_amount' => $order->total_amount,
                'formatted_total' => number_format($order->total_amount, 2) . ' ' . (app('settings')['currency'] ?? '₺'),
                'status' => $order->status,
                'status_text' => $order->status_text,
                'status_badge' => $order->status_badge,
                'payment_status' => $order->payment_status,
                'payment_status_text' => $order->payment_status_text,
                'payment_status_badge' => $order->payment_status_badge,
                'created_at' => $order->created_at->format('d.m.Y H:i'),
                'delivered_at' => $order->delivered_at ? $order->delivered_at->format('d.m.Y H:i') : null,
                'notes' => $order->notes,
                'product_data' => $order->product_data,
                'can_be_cancelled' => $order->canBeCancelled(),
                'product' => $order->product ? [
                    'id' => $order->product->id,
                    'product_name' => $order->product->product_name,
                    'description' => $order->product->description,
                    'is_digital' => $order->product->is_digital,
                    'image_url' => $order->product->image_url
                ] : null
            ];
            
            // Admin için user bilgilerini ekle
            if ($user->user_type == 1 && $order->user) {
                $orderData['user_name'] = $order->user->account_holder ?? $order->user->name;
                $orderData['user_email'] = $order->user->email;
            }
            
            return response()->json([
                'success' => true,
                'order' => $orderData
            ]);
        }
        
        return view('orders.show', compact('order'));
    }
    
    // API: Sipariş detayı (AJAX için)
    public function getOrderDetails($id)
    {
        $user = Auth::user();
        
        if ($user->user_type == 1) {
            $order = Order::with(['user', 'product'])->findOrFail($id);
        } else {
            $order = Order::where('user_id', $user->id)
                ->with(['product'])
                ->findOrFail($id);
        }
        
        return response()->json([
            'success' => true,
            'order' => [
                'id' => $order->id,
                'order_number' => $order->order_number,
                'product_name' => $order->product_name,
                'product_price' => $order->product_price,
                'quantity' => $order->quantity,
                'total_amount' => $order->total_amount,
                'formatted_total' => number_format($order->total_amount, 2) . ' ' . (app('settings')['currency'] ?? '₺'),
                'status' => $order->status,
                'status_text' => $order->status_text,
                'status_badge' => $order->status_badge,
                'payment_status' => $order->payment_status,
                'payment_status_text' => $order->payment_status_text,
                'payment_status_badge' => $order->payment_status_badge,
                'created_at' => $order->created_at->format('d.m.Y H:i'),
                'delivered_at' => $order->delivered_at ? $order->delivered_at->format('d.m.Y H:i') : null,
                'notes' => $order->notes,
                'product_data' => $order->product_data,
                'can_be_cancelled' => $order->canBeCancelled(),
                'product' => $order->product ? [
                    'id' => $order->product->id,
                    'product_name' => $order->product->product_name,
                    'description' => $order->product->description,
                    'is_digital' => $order->product->is_digital,
                    'image_url' => $order->product->image_url
                ] : null
            ]
        ]);
    }
    
    // Sipariş iptal et
    public function cancel($id)
    {
        $user = Auth::user();
        $order = Order::where('user_id', $user->id)->findOrFail($id);
        
        if (!$order->canBeCancelled()) {
            return response()->json([
                'success' => false,
                'message' => __('Order cannot be cancelled')
            ], 400);
        }
        
        DB::beginTransaction();
        try {
            // Refund balance if paid
            if ($order->payment_status === 'paid') {
                $user->increment('balance', $order->total_amount);
                
                // Log balance transaction
                $balanceBefore = $user->balance - $order->total_amount; // Çünkü zaten increment yapıldı
                \App\Models\BalanceLog::create([
                    'userId' => $user->id,
                    'amount' => $order->total_amount,
                    'operation_type' => 'refund',
                    'description' => "Order cancellation refund: {$order->order_number}",
                    'reference_id' => $order->id,
                    'reference_type' => 'order',
                    'balance_before' => $balanceBefore,
                    'balance' => $user->balance,
                    'added_user' => \Auth::id()
                ]);
                
                $order->update(['payment_status' => 'refunded']);
            }
            
            // Restore stock
            $order->product->increaseStock($order->quantity);
            
            // Update order status
            $order->update(['status' => 'cancelled']);
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => __('Order cancelled successfully')
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('Error cancelling order')
            ], 500);
        }
    }
    
    // Admin: Sipariş durumu güncelle
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:pending,processing,completed,cancelled',
            'notes' => 'nullable|string|max:1000'
        ]);
        
        $order = Order::with(['user', 'product'])->findOrFail($id);
        
        DB::beginTransaction();
        try {
            $oldStatus = $order->status;
            
            // Handle status changes
            if ($request->status === 'cancelled' && $oldStatus !== 'cancelled') {
                // Admin tarafından iptal edildiğinde bakiye iadesi yap
                if ($order->payment_status === 'paid') {
                    $balanceBefore = $order->user->balance;
                    $order->user->increment('balance', $order->total_amount);
                    $balanceAfter = $balanceBefore + $order->total_amount;
                    
                    // Log balance transaction
                    \App\Models\BalanceLog::create([
                        'userId' => $order->user_id,
                        'amount' => $order->total_amount,
                        'operation_type' => 'refund',
                        'description' => "Admin order cancellation refund: {$order->order_number}",
                        'reference_id' => $order->id,
                        'reference_type' => 'order',
                        'balance_before' => $balanceBefore,
                        'balance' => $balanceAfter,
                        'added_user' => \Auth::id()
                    ]);
                    
                    // Payment status'u refunded yap
                    $order->update(['payment_status' => 'refunded']);
                }
                
                // Stok iadesi yap
                if ($order->product) {
                    $order->product->increment('stock_quantity', $order->quantity);
                }
            }
            
            if ($request->status === 'completed' && $oldStatus !== 'completed') {
                // Dijital ürünler için otomatik teslimat
                if ($order->product && $order->product->is_digital) {
                    $this->processDigitalDelivery($order);
                } else {
                    $order->markAsDelivered();
                }
            }
            
            // Sipariş durumunu güncelle
            $order->update([
                'status' => $request->status,
                'notes' => $request->notes
            ]);
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => __('Order status updated successfully')
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            \Log::error('Order status update error: ' . $e->getMessage(), [
                'order_id' => $id,
                'status' => $request->status,
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => __('Error updating order status') . ': ' . $e->getMessage()
            ], 500);
        }
    }
    
    // Admin: DataTable API
    public function adminDataTable(Request $request)
    {
        try {
            $query = Order::with(['user', 'product']);
            
            // Filters
            if ($request->status) {
                $query->where('status', $request->status);
            }
            
            if ($request->payment_status) {
                $query->where('payment_status', $request->payment_status);
            }
            
            // Search
            if ($request->search && isset($request->search['value']) && $request->search['value']) {
                $searchValue = $request->search['value'];
                $query->where(function($q) use ($searchValue) {
                    $q->where('order_number', 'like', "%{$searchValue}%")
                      ->orWhere('product_name', 'like', "%{$searchValue}%")
                      ->orWhereHas('user', function($userQuery) use ($searchValue) {
                          $userQuery->where('account_holder', 'like', "%{$searchValue}%")
                                   ->orWhere('email', 'like', "%{$searchValue}%");
                      });
                });
            }
            
            $totalRecords = $query->count();
            
            // Ordering
            if ($request->order && isset($request->order[0])) {
                $orderColumnIndex = $request->order[0]['column'];
                $orderDirection = $request->order[0]['dir'];
                
                if (isset($request->columns[$orderColumnIndex]['name'])) {
                    $orderColumn = $request->columns[$orderColumnIndex]['name'];
                    $query->orderBy($orderColumn, $orderDirection);
                } else {
                    $query->orderBy('created_at', 'desc');
                }
            } else {
                $query->orderBy('created_at', 'desc');
            }
            
            // Pagination
            $start = $request->start ?? 0;
            $length = $request->length ?? 25;
            $orders = $query->skip($start)->take($length)->get();
            
            $data = $orders->map(function($order) {
                return [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'user_name' => $order->user->account_holder ?? $order->user->name,
                    'user_email' => $order->user->email,
                    'product_name' => $order->product_name,
                    'quantity' => $order->quantity,
                    'total_amount' => $order->formatted_total,
                    'status' => $order->status,
                    'status_text' => $order->status_text,
                    'status_badge' => $order->status_badge,
                    'payment_status' => $order->payment_status,
                    'payment_status_text' => $order->payment_status_text,
                    'payment_status_badge' => $order->payment_status_badge,
                    'created_at' => $order->created_at->format('d.m.Y H:i'),
                    'delivered_at' => $order->delivered_at ? $order->delivered_at->format('d.m.Y H:i') : null
                ];
            });
            
            return response()->json([
                'draw' => intval($request->draw ?? 1),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => $data
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'draw' => intval($request->draw ?? 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading data'
            ], 500);
        }
    }
    
    // Digital product delivery
    private function processDigitalDelivery($order)
    {
        $product = $order->product;
        $deliveryData = [];
        
        if ($product->digital_content) {
            // Process digital content delivery
            $deliveryData = [
                'delivery_method' => 'automatic',
                'delivered_at' => now()->toISOString(),
                'content' => $product->digital_content,
                'instructions' => $product->delivery_instructions
            ];
        }
        
        $order->markAsDelivered($deliveryData);
    }
}