<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\ProductCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        
        if ($user && $user->user_type == 1) {
            // Admin view
            return $this->adminIndex($request);
        }
        
        // User view - Ajax için
        if ($request->ajax()) {
            return $this->getProductsAjax($request);
        }
        
        // Get categories for filter
        $categories = ProductCategory::where('status', 1)->get();
        
        return view('products.index', compact('categories'));
    }
    
    // User: Ajax ürün listesi
    public function getProductsAjax(Request $request)
    {
        $page = $request->get('page', 1);
        $perPage = 12;
        
        $query = Product::active()
            ->inStock()
            ->with(['category'])
            ->orderBy('created_at', 'desc');
            
        // Filters
        if ($request->category_id) {
            $query->where('category_id', $request->category_id);
        }
        
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('product_name', 'like', "%{$request->search}%")
                  ->orWhere('description', 'like', "%{$request->search}%");
            });
        }
        
        if ($request->price_range) {
            switch ($request->price_range) {
                case 'low':
                    $query->where('price', '<=', 50);
                    break;
                case 'medium':
                    $query->whereBetween('price', [50, 200]);
                    break;
                case 'high':
                    $query->where('price', '>', 200);
                    break;
            }
        }
        
        $products = $query->paginate($perPage, ['*'], 'page', $page);
        
        // Format products data
        $formattedProducts = $products->getCollection()->map(function($product) {
            return [
                'id' => $product->id,
                'product_name' => $product->product_name,
                'description' => $product->description,
                'price' => $product->price,
                'formatted_price' => number_format($product->price, 2) . ' ' . (app('settings')['currency'] ?? '₺'),
                'image_url' => $product->image_url,
                'is_digital' => $product->is_digital,
                'stock_quantity' => $product->stock_quantity,
                'stock_status' => $product->stock_status,
                'stock_badge' => $product->stock_badge,
                'status' => $product->status,
                'category' => $product->category ? [
                    'id' => $product->category->id,
                    'name' => $product->category->name
                ] : null,
                'min_quantity' => $product->min_quantity,
                'max_quantity' => $product->max_quantity,
                'provider' => $product->provider
            ];
        });
        
        return response()->json([
            'success' => true,
            'products' => $formattedProducts,
            'has_more' => $products->hasMorePages(),
            'current_page' => $products->currentPage()
        ]);
    }
    
    // Admin: Ürün yönetimi
    public function adminIndex(Request $request)
    {
        // Statistics
        $stats = [
            'total' => Product::count(),
            'active' => Product::active()->count(),
            'digital' => Product::digital()->count(),
            'low_stock' => Product::where('is_digital', false)
                ->where('stock_quantity', '<=', 10)
                ->where('stock_quantity', '>', 0)
                ->count(),
            'out_of_stock' => Product::where('is_digital', false)
                ->where('stock_quantity', 0)
                ->count()
        ];
        
        return view('products.admin.index', compact('stats'));
    }
    
    public function show($id)
    {
        $product = Product::with('category')->findOrFail($id);
        
        // Check if user can view this product
        if (!$product->status && (!Auth::check() || Auth::user()->user_type != 1)) {
            abort(404);
        }
        
        // Admin için JSON response
        if (Auth::check() && Auth::user()->user_type == 1 && request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'product' => $product
            ]);
        }
        
        return view('products.show', compact('product'));
    }
    
    // Admin: DataTable API
    public function adminDataTable(Request $request)
    {
        try {
            $query = Product::with(['category']);
            
            // Filters
            if ($request->category_id) {
                $query->where('category_id', $request->category_id);
            }
            
            if ($request->status !== null && $request->status !== '') {
                $query->where('status', $request->status);
            }
            
            if ($request->type) {
                if ($request->type === 'digital') {
                    $query->where('is_digital', true);
                } elseif ($request->type === 'physical') {
                    $query->where('is_digital', false);
                }
            }
            
            // Search
            if ($request->search && isset($request->search['value']) && $request->search['value']) {
                $searchValue = $request->search['value'];
                $query->where(function($q) use ($searchValue) {
                    $q->where('product_name', 'like', "%{$searchValue}%")
                      ->orWhere('description', 'like', "%{$searchValue}%")
                      ->orWhere('provider', 'like', "%{$searchValue}%");
                });
            }
            
            $totalRecords = $query->count();
            
            // Ordering
            if ($request->order && isset($request->order[0])) {
                $orderColumnIndex = $request->order[0]['column'];
                $orderDirection = $request->order[0]['dir'];
                
                if (isset($request->columns[$orderColumnIndex]['name'])) {
                    $orderColumn = $request->columns[$orderColumnIndex]['name'];
                    $query->orderBy($orderColumn, $orderDirection);
                } else {
                    $query->orderBy('created_at', 'desc');
                }
            } else {
                $query->orderBy('created_at', 'desc');
            }
            
            // Pagination
            $start = $request->start ?? 0;
            $length = $request->length ?? 25;
            $products = $query->skip($start)->take($length)->get();
            
            $data = $products->map(function($product) {
                return [
                    'id' => $product->id,
                    'product_name' => $product->product_name,
                    'category_name' => $product->category->name ?? 'N/A',
                    'price' => $product->formatted_price,
                    'is_digital' => $product->is_digital,
                    'stock_quantity' => $product->stock_quantity,
                    'stock_status' => $product->stock_status,
                    'stock_badge' => $product->stock_badge,
                    'status' => $product->status,
                    'status_text' => $product->status_text,
                    'status_badge' => $product->status_badge,
                    'provider' => $product->provider,
                    'created_at' => $product->created_at->format('d.m.Y H:i')
                ];
            });
            
            return response()->json([
                'draw' => intval($request->draw ?? 1),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => $data
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'draw' => intval($request->draw ?? 1),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading data'
            ], 500);
        }
    }
    public function store(Request $request)
    {
        $validated = $request->validate([
            'product_name' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric',
            'image_url' => 'required|string',
            'category_id' => 'required|integer|exists:product_categories,id',
            'min_quantity' => 'required|integer',
            'max_quantity' => 'required|integer',
            'status' => 'required|boolean'
        ]);

        Product::create($validated);

        return response()->json(['success' => true, 'message' => 'Product added successfully']);
    }

    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'product_name' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'category_id' => 'required|integer|exists:product_categories,id',
            'is_digital' => 'required|boolean',
            'stock_quantity' => 'nullable|integer|min:0',
            'digital_content' => 'nullable|string',
            'auto_delivery' => 'nullable|boolean',
            'min_quantity' => 'required|integer|min:1',
            'max_quantity' => 'required|integer|min:1',
            'provider' => 'required|string|max:255',
            'status' => 'required|boolean',
            'image_url' => 'nullable|string'
        ]);

        $product = Product::findOrFail($id);
        
        // Auto delivery sadece dijital ürünler için
        if (!$validated['is_digital']) {
            $validated['auto_delivery'] = false;
            $validated['digital_content'] = null;
        }
        
        // Fiziksel ürünler için stock_quantity zorunlu
        if (!$validated['is_digital'] && !isset($validated['stock_quantity'])) {
            $validated['stock_quantity'] = 0;
        }
        
        // Image URL yoksa mevcut değeri koru
        if (!isset($validated['image_url']) || empty($validated['image_url'])) {
            $validated['image_url'] = $product->image_url;
        }
        
        $product->update($validated);

        return response()->json(['success' => true, 'message' => 'Product updated successfully']);
    }




    public function create()
{
    $categories = ProductCategory::where('status', 1)->get();
    $html = view('products.partials.form', compact('categories'))->render();
    return response()->json(['html' => $html]);
}

public function edit($id)
{
    $product = Product::findOrFail($id);
    $categories = ProductCategory::where('status', 1)->get();
    $html = view('products.partials.form', compact('product', 'categories'))->render();
    return response()->json(['html' => $html]);
}


    public function destroy($id)
    {
        $product = Product::findOrFail($id);
        $product->delete();

        return response()->json(['success' => true, 'message' => 'Product deleted successfully']);
    }

    public function getCategories()
    {
        $categories = ProductCategory::where('status', 1)->get(['id', 'name']);
        
        return response()->json([
            'success' => true,
            'categories' => $categories
        ]);
    }
}
