<?php

namespace App\Http\Controllers;

use App\Models\BalanceLog;
use App\Models\TransactionLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ProfitController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Kullanıcı kazanç sayfası
     */
    public function userIndex()
    {
        $user = Auth::user();
        
        // Temel istatistikler
        $stats = $this->getUserStats($user->id);
        
        // Haftalık veriler (son 7 gün)
        $weeklyData = $this->getWeeklyData($user->id);
        
        // Aylık veriler (son 12 ay)
        $monthlyData = $this->getMonthlyData($user->id);
        
        // Kazanç türlerine göre dağılım
        $earningsByType = $this->getEarningsByType($user->id);
        
        return view('profits.user', compact('stats', 'weeklyData', 'monthlyData', 'earningsByType'));
    }

    /**
     * Admin kazanç sayfası
     */
    public function adminIndex()
    {
        // Admin kontrolü
        if (Auth::user()->membership_type != 1) {
            return redirect('/home')->with('error', 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        // Genel istatistikler
        $stats = $this->getAdminStats();
        
        // Haftalık veriler
        $weeklyData = $this->getWeeklyDataAdmin();
        
        // Aylık veriler
        $monthlyData = $this->getMonthlyDataAdmin();
        
        // Kazanç türlerine göre dağılım
        $earningsByType = $this->getEarningsByTypeAdmin();
        
        // En çok kazanan kullanıcılar
        $topEarners = $this->getTopEarners();
        
        return view('profits.admin', compact('stats', 'weeklyData', 'monthlyData', 'earningsByType', 'topEarners'));
    }

    /**
     * Kullanıcı kazanç verileri API
     */
    public function getUserData(Request $request)
    {
        $user = Auth::user();
        $stats = $this->getUserStats($user->id);
        $weeklyData = $this->getWeeklyData($user->id);
        $monthlyData = $this->getMonthlyData($user->id);
        $earningsByType = $this->getEarningsByType($user->id);
        
        return response()->json([
            'success' => true,
            'stats' => $stats,
            'weeklyData' => $weeklyData,
            'monthlyData' => $monthlyData,
            'earningsByType' => $earningsByType
        ]);
    }

    /**
     * Admin kazanç verileri API
     */
    public function getAdminData(Request $request)
    {
        if (Auth::user()->membership_type != 1) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $stats = $this->getAdminStats();
        $weeklyData = $this->getWeeklyDataAdmin();
        $monthlyData = $this->getMonthlyDataAdmin();
        $earningsByType = $this->getEarningsByTypeAdmin();
        $topEarners = $this->getTopEarners();
        
        return response()->json([
            'success' => true,
            'stats' => $stats,
            'weeklyData' => $weeklyData,
            'monthlyData' => $monthlyData,
            'earningsByType' => $earningsByType,
            'topEarners' => $topEarners
        ]);
    }

    /**
     * Kullanıcı kazanç verileri API (eski)
     */
    public function getUserEarningsApi(Request $request)
    {
        $user = Auth::user();
        $period = $request->get('period', 'week');
        
        $query = BalanceLog::where('userId', $user->id)
            ->where('operation_type', 'credit')
            ->with(['task'])
            ->orderBy('created_at', 'desc');
        
        // Tarih filtresi
        switch ($period) {
            case 'week':
                $query->where('created_at', '>=', Carbon::now()->subWeek());
                break;
            case 'month':
                $query->where('created_at', '>=', Carbon::now()->subMonth());
                break;
            case 'year':
                $query->where('created_at', '>=', Carbon::now()->subYear());
                break;
        }
        
        $earnings = $query->paginate(20);
        
        return response()->json([
            'success' => true,
            'earnings' => $earnings
        ]);
    }

    /**
     * Admin kazanç verileri API
     */
    public function getAdminEarningsApi(Request $request)
    {
        if (Auth::user()->membership_type != 1) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }
        
        $period = $request->get('period', 'week');
        $type = $request->get('type', 'all');
        
        $query = BalanceLog::where('operation_type', 'credit')
            ->with(['user', 'task'])
            ->orderBy('created_at', 'desc');
        
        // Tarih filtresi
        switch ($period) {
            case 'week':
                $query->where('created_at', '>=', Carbon::now()->subWeek());
                break;
            case 'month':
                $query->where('created_at', '>=', Carbon::now()->subMonth());
                break;
            case 'year':
                $query->where('created_at', '>=', Carbon::now()->subYear());
                break;
        }
        
        // Tür filtresi
        if ($type != 'all') {
            $query->where('reference_type', $type);
        }
        
        $earnings = $query->paginate(50);
        
        return response()->json([
            'success' => true,
            'earnings' => $earnings
        ]);
    }

    /**
     * Kullanıcı istatistikleri
     */
    private function getUserStats($userId)
    {
        $user = User::find($userId);
        
        return [
            'current_balance' => $user->balance ?? 0,
            'total_earnings' => BalanceLog::where('userId', $userId)
                ->where('operation_type', 'credit')
                ->sum('amount') ?? 0,
            'monthly_earnings' => BalanceLog::where('userId', $userId)
                ->where('operation_type', 'credit')
                ->where('created_at', '>=', Carbon::now()->startOfMonth())
                ->sum('amount') ?? 0,
            'weekly_earnings' => BalanceLog::where('userId', $userId)
                ->where('operation_type', 'credit')
                ->where('created_at', '>=', Carbon::now()->startOfWeek())
                ->sum('amount') ?? 0,
            'daily_earnings' => BalanceLog::where('userId', $userId)
                ->where('operation_type', 'credit')
                ->whereDate('created_at', Carbon::today())
                ->sum('amount') ?? 0,
            'completed_tasks' => BalanceLog::where('userId', $userId)
                ->where('operation_type', 'credit')
                ->distinct('TaskId')
                ->count('TaskId') ?? 0
        ];
    }

    /**
     * Admin istatistikleri
     */
    private function getAdminStats()
    {
        return [
            'total_users' => User::where('membership_type', 2)->count(),
            'total_earnings' => BalanceLog::where('operation_type', 'credit')->sum('amount') ?? 0,
            'today_earnings' => BalanceLog::where('operation_type', 'credit')
                ->whereDate('created_at', Carbon::today())
                ->sum('amount') ?? 0,
            'monthly_average' => BalanceLog::where('operation_type', 'credit')
                ->where('created_at', '>=', Carbon::now()->subMonths(12))
                ->sum('amount') / 12 ?? 0,
            'active_users' => User::where('membership_type', 2)
                ->where('is_online', 1)
                ->count(),
            'total_tasks_completed' => BalanceLog::where('operation_type', 'credit')
                ->whereNotNull('TaskId')
                ->count()
        ];
    }

    /**
     * Haftalık veriler (kullanıcı)
     */
    private function getWeeklyData($userId)
    {
        $labels = [];
        $data = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $earnings = BalanceLog::where('userId', $userId)
                ->where('operation_type', 'credit')
                ->whereDate('created_at', $date)
                ->sum('amount') ?? 0;
            
            $labels[] = $date->format('M d');
            $data[] = floatval($earnings);
        }
        return [
            'labels' => $labels,
            'data' => $data
        ];
    }

    /**
     * Haftalık veriler (admin)
     */
    private function getWeeklyDataAdmin()
    {
        $labels = [];
        $data = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $earnings = BalanceLog::where('operation_type', 'credit')
                ->whereDate('created_at', $date)
                ->sum('amount') ?? 0;
            
            $labels[] = $date->format('M d');
            $data[] = floatval($earnings);
        }
        return [
            'labels' => $labels,
            'data' => $data
        ];
    }

    /**
     * Aylık veriler (kullanıcı)
     */
    private function getMonthlyData($userId)
    {
        $labels = [];
        $data = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = Carbon::now()->subMonths($i);
            $earnings = BalanceLog::where('userId', $userId)
                ->where('operation_type', 'credit')
                ->whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month)
                ->sum('amount') ?? 0;
            
            $labels[] = $date->format('M Y');
            $data[] = floatval($earnings);
        }
        return [
            'labels' => $labels,
            'data' => $data
        ];
    }

    /**
     * Aylık veriler (admin)
     */
    private function getMonthlyDataAdmin()
    {
        $labels = [];
        $data = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = Carbon::now()->subMonths($i);
            $earnings = BalanceLog::where('operation_type', 'credit')
                ->whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month)
                ->sum('amount') ?? 0;
            
            $labels[] = $date->format('M Y');
            $data[] = floatval($earnings);
        }
        return [
            'labels' => $labels,
            'data' => $data
        ];
    }

    /**
     * Kazanç türlerine göre dağılım (kullanıcı)
     */
    private function getEarningsByType($userId)
    {
        $socialMedia = BalanceLog::where('userId', $userId)
            ->where('operation_type', 'credit')
            ->where('reference_type', 'social_media')
            ->sum('amount') ?? 0;
        
        $websiteVisit = BalanceLog::where('userId', $userId)
            ->where('operation_type', 'credit')
            ->where('reference_type', 'website_visit')
            ->sum('amount') ?? 0;
        
        $youtubeWatch = BalanceLog::where('userId', $userId)
            ->where('operation_type', 'credit')
            ->where('reference_type', 'youtube_watch')
            ->sum('amount') ?? 0;
        
        return [
            'labels' => ['Social Media', 'Website Visit', 'YouTube Watch'],
            'data' => [floatval($socialMedia), floatval($websiteVisit), floatval($youtubeWatch)]
        ];
    }

    /**
     * Kazanç türlerine göre dağılım (admin)
     */
    private function getEarningsByTypeAdmin()
    {
        $socialMedia = BalanceLog::where('operation_type', 'credit')
            ->where('reference_type', 'social_media')
            ->sum('amount') ?? 0;
        
        $websiteVisit = BalanceLog::where('operation_type', 'credit')
            ->where('reference_type', 'website_visit')
            ->sum('amount') ?? 0;
        
        $youtubeWatch = BalanceLog::where('operation_type', 'credit')
            ->where('reference_type', 'youtube_watch')
            ->sum('amount') ?? 0;
        
        return [
            'labels' => ['Social Media', 'Website Visit', 'YouTube Watch'],
            'data' => [floatval($socialMedia), floatval($websiteVisit), floatval($youtubeWatch)]
        ];
    }

    /**
     * En çok kazanan kullanıcılar
     */
    private function getTopEarners()
    {
        return DB::table('balanceLog')
            ->select('users.account_holder', DB::raw('SUM(balanceLog.amount) as total_earnings'))
            ->where('balanceLog.operation_type', 'credit')
            ->join('users', 'balanceLog.userId', '=', 'users.id')
            ->groupBy('users.id', 'users.account_holder')
            ->orderBy('total_earnings', 'desc')
            ->limit(10)
            ->get();
    }
}