<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SettingsController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'isAdmin']);
    }

    public function index()
    {
        try {
            $settingsModel = Setting::first();
            if (!$settingsModel) {
                $settingsModel = new Setting();
            }
            
            // Debug için
            \Log::info('Settings data:', $settingsModel ? $settingsModel->toArray() : ['no data']);
            
            return view('admin.settings.index', ['settingsModel' => $settingsModel]);
        } catch (\Exception $e) {
            \Log::error('Settings index error: ' . $e->getMessage());
            $settingsModel = new Setting();
            return view('admin.settings.index', ['settingsModel' => $settingsModel]);
        }
    }

    public function update(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'site_name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'slogan' => 'nullable|string|max:255',
                'keywords' => 'nullable|string|max:1000',
                'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
                'favicon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,ico|max:1024',
                'komisyon' => 'nullable|numeric|min:0|max:100',
                'odeme' => 'nullable|string',
                'sms_dogrulama' => 'nullable|boolean',
                'mail_dogrulama' => 'nullable|boolean',
                'gunluk_bonus' => 'nullable|numeric|min:0',
                'min_bakiyecekim' => 'nullable|numeric|min:0',
                'ref' => 'nullable|numeric|min:0',
                'gold' => 'nullable|numeric|min:0',
                'platin' => 'nullable|numeric|min:0',
                'silver' => 'nullable|numeric|min:0',
                'gold_bonus' => 'nullable|numeric|min:0',
                'silver_bonus' => 'nullable|numeric|min:0',
                'platin_bonus' => 'nullable|numeric|min:0',
                'pbirim' => 'nullable|string|max:10',
                'currency' => 'nullable|string|max:10',
                'recaptcha_site_key' => 'nullable|string|max:500',
                'recaptcha_secret_key' => 'nullable|string|max:500',
                'recaptcha_enabled' => 'nullable|boolean',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $settings = Setting::first();
            if (!$settings) {
                $settings = new Setting();
            }

            // Boş değerleri varsayılan değerlerle değiştir
            $data = $request->all();
            $data['keywords'] = $data['keywords'] ?? '';
            $data['description'] = $data['description'] ?? '';
            $data['slogan'] = $data['slogan'] ?? '';
            $data['odeme'] = $data['odeme'] ?? '';
            $data['pbirim'] = $data['pbirim'] ?? 'TL';
            // pbirim değeri currency alanına da kaydedilsin
            $data['currency'] = $data['pbirim'];
            
            // Dosya yükleme işlemleri
            if ($request->hasFile('logo')) {
                $logoFile = $request->file('logo');
                $logoName = 'logo_' . time() . '.' . $logoFile->getClientOriginalExtension();
                $logoFile->move(public_path('assets/images'), $logoName);
                $data['logo'] = $logoName;
            }
            
            if ($request->hasFile('favicon')) {
                $faviconFile = $request->file('favicon');
                $faviconName = 'favicon_' . time() . '.' . $faviconFile->getClientOriginalExtension();
                $faviconFile->move(public_path('assets/images'), $faviconName);
                $data['favicon'] = $faviconName;
            }
            
            // Checkbox değerleri
            $data['sms_dogrulama'] = $request->has('sms_dogrulama') ? 1 : 0;
            $data['mail_dogrulama'] = $request->has('mail_dogrulama') ? 1 : 0;
            $data['recaptcha_enabled'] = $request->has('recaptcha_enabled') ? 1 : 0;
            
            // Numeric değerler
            $numericFields = ['komisyon', 'gunluk_bonus', 'min_bakiyecekim', 'ref', 'gold', 'platin', 'silver', 'gold_bonus', 'silver_bonus', 'platin_bonus'];
            foreach ($numericFields as $field) {
                $data[$field] = $data[$field] ?? 0;
            }

            $settings->fill($data);
            $settings->save();

            return response()->json([
                'success' => true,
                'message' => __('Settings updated successfully!')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while updating settings: ') . $e->getMessage()
            ], 500);
        }
    }
}