<?php
namespace App\Http\Controllers;

use App\Models\SocialMediaAccount;
use App\Models\SocialPlatform;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SocialMediaAccountController extends Controller
{
    // User Methods
    public function userIndex()
    {
        $user = Auth::user();
        return view('user.social_accounts.index', compact('user'));
    }

    public function userSearch(Request $request)
    {
        $search = $request->input('search');
        $status = $request->input('status');
        $page = $request->input('page', 1);
        $perPage = $request->input('per_page', 12);

        $query = SocialMediaAccount::with('platform')
            ->where('user_id', Auth::id());

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('username', 'like', "%{$search}%")
                  ->orWhereHas('platform', function($platformQuery) use ($search) {
                      $platformQuery->where('platform', 'like', "%{$search}%");
                  });
            });
        }

        if ($status !== null && $status !== '') {
            $query->where('status', $status);
        }

        $accounts = $query->orderBy('created_at', 'desc')
                         ->paginate($perPage, ['*'], 'page', $page);

        // Her hesap için platform bilgilerinin yüklendiğinden emin ol
        $accounts->getCollection()->transform(function ($account) {
            // Eğer platform yüklenmemişse varsayılan değerler ata
            if (!$account->platform) {
                $account->platform = (object) [
                    'id' => 0,
                    'platform' => 'Unknown Platform',
                    'icon' => 'placeholder.png'
                ];
            }
            
            // Görev yapılmış mı kontrol et
            $account->has_completed_tasks = $this->hasCompletedTasks($account);
            
            return $account;
        });

        return response()->json([
            'success' => true,
            'data' => $accounts->items(),
            'pagination' => [
                'current_page' => $accounts->currentPage(),
                'last_page' => $accounts->lastPage(),
                'per_page' => $accounts->perPage(),
                'total' => $accounts->total(),
                'from' => $accounts->firstItem(),
                'to' => $accounts->lastItem()
            ]
        ]);
    }

    // Admin Methods
    public function adminIndex()
    {
        $user = Auth::user();
        return view('admin.social_accounts.index', compact('user'));
    }

    public function adminSearch(Request $request)
    {
        $search = $request->input('search');
        $status = $request->input('status');
        $page = $request->input('page', 1);
        $perPage = $request->input('per_page', 12);

        $query = SocialMediaAccount::with(['platform', 'user']);

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('username', 'like', "%{$search}%")
                  ->orWhereHas('platform', function($platformQuery) use ($search) {
                      $platformQuery->where('platform', 'like', "%{$search}%");
                  })
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('account_holder', 'like', "%{$search}%")
                               ->orWhere('username', 'like', "%{$search}%");
                  });
            });
        }

        if ($status !== null && $status !== '') {
            $query->where('status', $status);
        }

        $accounts = $query->orderBy('created_at', 'desc')
                         ->paginate($perPage, ['*'], 'page', $page);

        // Her hesap için user ve platform bilgilerinin yüklendiğinden emin ol
        $accounts->getCollection()->transform(function ($account) {
            // Eğer user yüklenmemişse varsayılan değerler ata
            if (!$account->user) {
                $account->user = (object) [
                    'id' => 0,
                    'account_holder' => 'Unknown User',
                    'username' => 'unknown'
                ];
            }
            
            // Eğer platform yüklenmemişse varsayılan değerler ata
            if (!$account->platform) {
                $account->platform = (object) [
                    'id' => 0,
                    'platform' => 'Unknown Platform',
                    'icon' => 'placeholder.png'
                ];
            }
            
            return $account;
        });

        return response()->json([
            'success' => true,
            'data' => $accounts->items(),
            'pagination' => [
                'current_page' => $accounts->currentPage(),
                'last_page' => $accounts->lastPage(),
                'per_page' => $accounts->perPage(),
                'total' => $accounts->total(),
                'from' => $accounts->firstItem(),
                'to' => $accounts->lastItem()
            ]
        ]);
    }
    public function store(Request $request)
    {
        // Platform bilgisini al
        $platform = SocialPlatform::find($request->platform_id);
        
        $validationRules = [
            'platform_id' => 'required|exists:socialplatforms,id',
            'username' => 'required|string|max:255',
        ];
        
        // Google Maps için özel validation
        if ($platform && (stripos($platform->platform, 'google') !== false && stripos($platform->platform, 'maps') !== false)) {
            $validationRules['username'] = [
                'required',
                'string',
                'max:500',
                'regex:/^https:\/\/www\.google\.com\/maps\/contrib\/\d+\/reviews\/@[\d\.,\-]+\/data=.*$/'
            ];
        }
        
        $request->validate($validationRules, [
            'username.regex' => __('Google Maps link must be a valid public local guide profile link. Example: https://www.google.com/maps/contrib/112743908407009889044/reviews/@40.4557731,29.9214183,7z/data=!3m1!4b1!4m3!8m2!3m1!1e1?entry=ttu&g_ep=...')
        ]);

        // Platform için zaten hesap var mı kontrol et
        $existingAccountForPlatform = SocialMediaAccount::where('user_id', Auth::id())
            ->where('platform_id', $request->platform_id)
            ->whereIn('status', [0, 1, 2]) // 0: Beklemede, 1: Onaylı, 2: Reddedildi
            ->first();

        if ($existingAccountForPlatform) {
            return response()->json([
                'success' => false,
                'message' => __('You already have an account for this platform.')
            ], 422);
        }

        // Username zaten kullanılıyor mu kontrol et
        $existingAccountByUsername = SocialMediaAccount::where('username', $request->username)
            ->where('platform_id', $request->platform_id)
            ->first();

        if ($existingAccountByUsername) {
            return response()->json([
                'success' => false,
                'message' => __('This username is already taken for this platform.')
            ], 422);
        }

        $account = SocialMediaAccount::create([
            'user_id' => Auth::id(),
            'platform_id' => $request->platform_id,
            'username' => $request->username,
            'status' => 0 // Beklemede
        ]);

        return response()->json([
            'success' => true,
            'message' => __('Account added successfully. Waiting for approval.'),
            'account' => $account->load('platform')
        ]);
    }

    public function approve($id)
    {
        $account = SocialMediaAccount::findOrFail($id);
        $account->status = 1; // Onaylı
        $account->approved_at = now();
        $account->save();

        return response()->json([
            'success' => true,
            'message' => __('Account approved successfully.'),
            'account' => $account->load(['platform', 'user'])
        ]);
    }

    public function reject($id)
    {
        $account = SocialMediaAccount::findOrFail($id);
        $account->status = 2; // Reddedildi
        $account->rejected_at = now();
        $account->save();

        return response()->json([
            'success' => true,
            'message' => __('Account rejected successfully.'),
            'account' => $account->load(['platform', 'user'])
        ]);
    }

    public function destroy($id)
    {
        try {
            $account = SocialMediaAccount::where('user_id', Auth::id())->findOrFail($id);
            
            // Görev yapılmış hesapları kontrol et
            $hasCompletedTasks = $this->hasCompletedTasks($account);
            
            if ($hasCompletedTasks) {
                return response()->json([
                    'success' => false,
                    'message' => __('Cannot delete account with completed tasks.')
                ], 422);
            }

            $account->delete();

            return response()->json([
                'success' => true,
                'message' => __('Account deleted successfully.')
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while deleting the account.')
            ], 500);
        }
    }

    public function adminDestroy($id)
    {
        $account = SocialMediaAccount::findOrFail($id);
        
        // Görev yapılmış hesapları kontrol et
        $hasCompletedTasks = $this->hasCompletedTasks($account);
        
        if ($hasCompletedTasks) {
            return response()->json([
                'success' => false,
                'message' => __('Cannot delete account with completed tasks.')
            ], 422);
        }

        $account->delete();

        return response()->json([
            'success' => true,
            'message' => __('Account deleted successfully.')
        ]);
    }

    // Platforms listesi için helper metot
    public function getPlatforms()
    {
        $platforms = SocialPlatform::where('status', 1)
            ->select('id', 'platform', 'icon')
            ->orderBy('platform')
            ->get();

        return response()->json([
            'success' => true,
            'platforms' => $platforms
        ]);
    }

    // Hesabın tamamlanmış görevleri olup olmadığını kontrol et
    private function hasCompletedTasks(SocialMediaAccount $account)
    {
        // TaskProof tablosunda bu kullanıcının bu platform için onaylanmış görevleri var mı kontrol et
        $completedTasksCount = DB::table('TaskProof as tp')
            ->join('tasks as t', 'tp.TaskId', '=', 't.id')
            ->where('tp.UserId', $account->user_id)
            ->where('t.category_id', $account->platform_id)
            ->where('tp.Status', 1) // Onaylanmış görevler
            ->count();

        return $completedTasksCount > 0;
    }
}
