<?php

namespace App\Http\Controllers;

use App\Models\SocialPlatform;
use App\Models\Task;
use App\Models\TaskProof;
use App\Models\User;
use App\Models\UserSession;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class SocialPlatformController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        return view('social-platforms.index', compact('user'));
    }

    public function show(SocialPlatform $socialPlatform)
    {
        // Görev sayısını ekle
        $socialPlatform->tasks_count = Task::where('category_id', $socialPlatform->id)->count();
        return response()->json($socialPlatform);
    }


    public function store(Request $request)
    {
        $request->validate([
            'platform' => 'required|string|max:255',
            'icon' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'nullable|string',
            'status' => 'required|boolean',
        ]);

        $iconName = $this->uploadIcon($request->file('icon'));

        $platform = SocialPlatform::create([
            'platform' => $request->platform,
            'icon' => $iconName,
            'description' => $request->description,
            'status' => $request->status
        ]);

        return response()->json(['success' => true, 'message' => __('Platform added successfully!'), 'platform' => $platform]);
    }

    public function update(Request $request, SocialPlatform $socialPlatform)
    {
        $request->validate([
            'platform' => 'required|string|max:255',
            'icon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'description' => 'nullable|string',
            'status' => 'required|boolean',
        ]);

        if ($request->hasFile('icon')) {
            // Eski ikonu sil
            if ($socialPlatform->icon && File::exists(public_path('assets/images/app/' . $socialPlatform->icon))) {
                File::delete(public_path('assets/images/app/' . $socialPlatform->icon));
            }
            // Yeni ikonu kaydet
            $iconName = $this->uploadIcon($request->file('icon'));
        } else {
            $iconName = $socialPlatform->icon;
        }

        $socialPlatform->update([
            'platform' => $request->platform,
            'icon' => $iconName,
            'description' => $request->description,
            'status' => $request->status
        ]);

        return response()->json(['success' => true, 'message' => __('Platform updated successfully!'), 'platform' => $socialPlatform]);
    }

    public function destroy(SocialPlatform $socialPlatform)
    {
        // Platforma kayıtlı görev var mı kontrol et
        $hasActiveTasks = Task::where('category_id', $socialPlatform->id)->exists();
        
        if ($hasActiveTasks) {
            return response()->json([
                'success' => false, 
                'message' => __('Cannot delete platform. There are tasks associated with this platform.')
            ], 422);
        }

        // Eski ikonu sil
        if ($socialPlatform->icon && File::exists(public_path('assets/images/app/' . $socialPlatform->icon))) {
            File::delete(public_path('assets/images/app/' . $socialPlatform->icon));
        }

        $socialPlatform->delete();

        return response()->json(['success' => true, 'message' => __('Platform deleted successfully!')]);
    }

    public function search(Request $request)
    {
        $search = $request->input('search');
        $status = $request->input('status');
        $page = $request->input('page', 1);
        $perPage = $request->input('per_page', 12);

        $query = SocialPlatform::query();

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('platform', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        if ($status !== null && $status !== '') {
            $query->where('status', $status);
        }

        $platforms = $query->orderBy('created_at', 'desc')
                          ->paginate($perPage, ['*'], 'page', $page);

        // Her platform için görev sayısını ekle
        $platforms->getCollection()->transform(function ($platform) {
            $platform->tasks_count = Task::where('category_id', $platform->id)->count();
            return $platform;
        });

        return response()->json([
            'success' => true,
            'data' => $platforms->items(),
            'pagination' => [
                'current_page' => $platforms->currentPage(),
                'last_page' => $platforms->lastPage(),
                'per_page' => $platforms->perPage(),
                'total' => $platforms->total(),
                'from' => $platforms->firstItem(),
                'to' => $platforms->lastItem()
            ]
        ]);
    }

    private function uploadIcon($icon)
    {
        $iconName = Str::random(10) . '.' . $icon->getClientOriginalExtension();
        $icon->move(public_path('assets/images/app'), $iconName);
        return $iconName;
    }
}
