<?php

namespace App\Http\Controllers;

use App\Models\UserSession;
use Illuminate\Support\Facades\Auth;
use App\Models\SocialPlatform;
use Illuminate\Support\Facades\Cache;
use App\Models\Task;
use App\Models\User;
use App\Models\Country;
use App\Models\City;
use App\Models\WebsiteVisitTask;
use App\Models\WatchTask;
use App\Models\BalanceLog;
use App\Models\TransactionLog;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TaskController extends Controller
{
    // Display a listing of the resource.
    public function index()
    {
        // Clear cache first to avoid old data issues
        Cache::forget('tasks_index_' . Auth::id());
        
        // Kullanıcının kendi görevlerini getir (Admin ise tüm görevler)
        $query = Task::with([
            'proofs' => function($query) {
                $query->where('status', 1);
            },
            'socialPlatform:id,platform,icon,description',
            'creator:id,account_holder,email,user_avatar',
            'city:id,name',
            'country:code,name'
        ])
        ->select('id', 'name', 'link', 'description', 'category_id', 'image', 'reward', 
                'limit', 'daily_limit', 'user_id', 'city', 'country_id', 'status', 'gender', 
                'button_text', 'proof_number', 'created_at');

        // Admin değilse sadece kendi görevlerini göster
        if (Auth::user()->user_type != 1) {
            $query->where('user_id', Auth::id());
        }

        $tasks = $query->orderBy('created_at', 'desc')->get();
        
        // Her task için completion count hesapla
        foreach ($tasks as $task) {
            $task->completion_count = $task->proofs->count();
            $task->completion_percentage = $task->limit > 0 ? ($task->completion_count / $task->limit) * 100 : 0;
        }
    
        $user = Auth::user();
        $platforms = Cache::remember('active_platforms', 600, function () {
            return SocialPlatform::active()->select('id', 'platform', 'icon', 'description')->get();
        });
        $countries = Cache::remember('all_countries', 3600, function () {
            return Country::select('code', 'name')->get();
        });
        $cities = Cache::remember('all_cities', 3600, function () {
            return City::select('id', 'name', 'country_id')->get();
        });
        
        return view('tasks.index', compact('tasks', 'user', 'platforms','countries','cities'));
    }

    // Show the form for creating a new resource.
    public function create()
    {
        return view('tasks.create');
    }

    // Store a newly created resource in storage.
    public function store(Request $request)
    {
        $messages = [
            'name.required' => __('Task name is required.'),
            'link.required' => __('Task link must be a valid URL.'),
            'link.url' => __('Task link must be a valid URL.'),
            'description.required' => __('Task description is required.'),
            'category_id.required' => __('Task category is required.'),
            'image.required' => __('Task image is required.'),
            'image.image' => __('Task image must be an image file.'),
            'image.mimes' => __('Task image must be a file of type: jpeg, png, jpg, gif, svg.'),
            'image.max' => __('Task image must not be greater than 2MB.'),
            'reward.required' => __('Task reward is required.'),
            'limit.required' => __('Task limit is required.'),
            'user_limit.required' => __('Task user limit is required.'),
            'country_id.required' => __('Task country is required.'),
            'city.required' => __('Task city is required.'),
            'gender.required' => __('Task gender is required.'),
            'button_text.required' => __('Task button text is required.'),
            'proof_number.required' => __('Task proof number is required.'),
            'daily_limit.required' => __('Task daily limit is required.'),
            'daily_limit.integer' => __('Task daily limit must be an integer.'),
            'daily_limit.numeric' => __('Task daily limit must be a number.'),
            

            // Add other messages as necessary
        ];
    
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'link' => 'required|url',
            'description' => 'required|string|max:1000',
            'category_id' => 'required|integer',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'reward' => 'required|numeric',
            'limit' => 'required|integer',
            'user_limit' => 'required|integer',
            'country_id' => 'required|integer',
            'city' => 'nullable|integer',
            'gender' => 'required|integer',
            'button_text' => 'required|string|max:500',
            'proof_number' => 'required|integer',
            'daily_limit' => 'nullable|integer',
          
        ], $messages);

        $user = Auth::user();
        
        // Görev maliyetini hesapla (reward * limit)
        $taskCost = $validated['reward'] * $validated['limit'];
        
        // Kullanıcının bakiyesini kontrol et
        if ($user->balance < $taskCost) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => __('Insufficient balance. Required: :cost₺, Available: :balance₺', [
                        'cost' => number_format($taskCost, 2),
                        'balance' => number_format($user->balance, 2)
                    ])
                ], 400);
            }
            
            return redirect()->back()->with('error', __('Insufficient balance. Required: :cost₺, Available: :balance₺', [
                'cost' => number_format($taskCost, 2),
                'balance' => number_format($user->balance, 2)
            ]))->withInput();
        }

        try {
            DB::beginTransaction();

            // Görevi oluştur
            $task = new Task($validated);
            $task->user_id = $user->id; 
            $task->status = '2'; // Onay bekliyor
            $task->gcode = "0";
            $task->pin = "0";
            $task->type = "1";
        
            if ($request->hasFile('image')) {
                $imageName = time().'.'.$request->image->extension();  
                $request->image->move(public_path('assets/images/tasks'), $imageName);
                $task->image = $imageName;
            }
        
            $task->save();

            // Kullanıcının bakiyesinden görev maliyetini düş
            $balanceBefore = $user->balance;
            $user->decrement('balance', $taskCost);
            $balanceAfter = $user->fresh()->balance;

            // Balance log kaydı oluştur
            BalanceLog::create([
                'user_id' => $user->id,
                'userId' => $user->id, // Eski format için
                'TaskId' => $task->id,
                'amount' => $taskCost,
                'balance' => $taskCost,
                'balance_before' => $balanceBefore,
                'balance_after' => $balanceAfter,
                'operation_type' => 'debit',
                'type' => 'task_creation',
                'description' => __('Task creation cost for ":task_name"', ['task_name' => $task->name]),
                'status' => 'completed',
                'reference_type' => 'task',
                'reference_id' => $task->id,
                'added_user' => $user->id
            ]);

            // Transaction log kaydı oluştur
            TransactionLog::create([
                'user_id' => $user->id,
                'type' => 'task_creation',
                'amount' => $taskCost,
                'description' => __('Task creation cost for ":task_name" (Reward: :reward₺ × Limit: :limit)', [
                    'task_name' => $task->name,
                    'reward' => number_format($validated['reward'], 2),
                    'limit' => $validated['limit']
                ]),
                'operation_type' => 'debit',
                'reference_type' => 'task',
                'reference_id' => $task->id,
                'balance_before' => $balanceBefore,
                'balance_after' => $balanceAfter
            ]);

            // Bildirim gönder
            $notificationService = app(NotificationService::class);
            $notificationService->notifyTaskCreated($task, $user);

            DB::commit();

            // Cache'i temizle
            Cache::forget('tasks_index_' . $user->id);

            // AJAX request ise JSON döndür
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => __('Task created successfully! Cost :cost₺ has been deducted from your balance. Task is waiting for admin approval.', [
                        'cost' => number_format($taskCost, 2)
                    ]),
                    'task' => $task,
                    'new_balance' => number_format($balanceAfter, 2)
                ]);
            }

            return redirect()->route('tasks.index')->with('success', __('Task created successfully! Cost :cost₺ has been deducted from your balance.', [
                'cost' => number_format($taskCost, 2)
            ]));

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Task creation error: ' . $e->getMessage());

            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => __('An error occurred while creating the task. Please try again.')
                ], 500);
            }

            return redirect()->back()->with('error', __('An error occurred while creating the task. Please try again.'))->withInput();
        }
    }
    

    public function getCitiesByCountry($countryCode)
    {
        try {
          
            
            // country_code ile şehirleri getir (integer olarak cast et)
            $cities = City::where('country_id', (int)$countryCode)->select('id', 'name')->get();
            
          
            
            return response()->json($cities);
        } catch (\Exception $e) {
          
            return response()->json([], 500);
        }
    }

    // Display the specified resource.
    public function show(Request $request, Task $task = null)
    {
        // Eğer task ID'si POST ile geliyorsa
        if ($request->isMethod('post') && $request->has('task_id')) {
            $task = Task::findOrFail($request->task_id);
        }
        
        // Eğer hala task yoksa, ID'yi URL'den al
        if (!$task && $request->has('id')) {
            $task = Task::findOrFail($request->id);
        }
        
        if (!$task) {
            abort(404, 'Task not found');
        }

        // Task ile ilgili verileri yükle
        $task = Task::with(['proofs' => function($query) {
            $query->select('TaskId', DB::raw('count(*) as total'))
                  ->where('status', 1)
                  ->groupBy('TaskId');
        }])
        ->leftJoin('socialplatforms', 'tasks.category_id', '=', 'socialplatforms.id')
        ->leftJoin('users', 'tasks.user_id', '=', 'users.id')
        ->leftJoin('cities', 'tasks.city', '=', 'cities.id')
        ->leftJoin('countries', 'tasks.country_id', '=', 'countries.code')
        ->select(
            'tasks.*', 
            'socialplatforms.platform as PlatformName',
            'socialplatforms.icon as PlatformIcon',
            'socialplatforms.description as PlatformDescription',
            'users.account_holder as UserName',
            'users.email as UserEmail',
            'users.user_avatar as UserAvatar',
            'cities.name as CityName',
            'countries.name as CountryName'
        )
        ->where('tasks.id', $task->id)
        ->first();

        // Kullanıcının IP'sinden konum bilgisi al (basit bir örnek)
        $response = [
            'IsoCode' => $task->city ?? 0 // Varsayılan olarak task'ın şehrini kullan
        ];

        // Proof sayısını hesapla
        $ProofCount = $task->proofs->isNotEmpty() ? $task->proofs->first()->total : 0;
        
        // Pending proof kontrolü
        $pendingProof = false;
        if (Auth::check()) {
            $pendingProof = DB::table('TaskProof')
                ->where('TaskId', $task->id)
                ->where('UserId', Auth::id())
                ->where('status', 0)
                ->exists();
        }

        // Settings
        $settings = [
            'currency' => '₺' // Varsayılan para birimi
        ];

        return view('tasks.show', compact('task', 'response', 'ProofCount', 'pendingProof', 'settings'));
    }

    // Show the form for editing the specified resource.
    public function edit(Task $task)
    {
        try {
            // AJAX request için JSON döndür - sadece AJAX için
            if (request()->expectsJson() || request()->ajax()) {
                // Task'ı tüm gerekli alanlarla birlikte yükle
                $task = Task::with(['socialPlatform', 'creator', 'city', 'country'])
                    ->find($task->id);
                
                if (!$task) {
                    return response()->json(['error' => 'Task not found'], 404);
                }
                
                return response()->json($task);
            }
            
            // Normal request için 404 döndür çünkü edit view'ı yok
            abort(404, 'Edit view not available');
        } catch (\Exception $e) {
            Log::error('Task edit error: ' . $e->getMessage());
            
            if (request()->expectsJson() || request()->ajax()) {
                return response()->json(['error' => 'An error occurred while fetching task data'], 500);
            }
            
            abort(404);
        }
    }

    // Update the specified resource in storage.
    public function update(Request $request, Task $task)
    {
        try {
            $messages = [
                'name.required' => __('Task name is required.'),
                'link.required' => __('Task link is required.'),
                'link.url' => __('Task link must be a valid URL.'),
                'description.required' => __('Task description is required.'),
                'category_id.required' => __('Task category is required.'),
                'reward.required' => __('Task reward is required.'),
                'limit.required' => __('Task limit is required.'),
                'user_limit.required' => __('Task user limit is required.'),
                'country_id.required' => __('Task country is required.'),
                'gender.required' => __('Task gender is required.'),
                'button_text.required' => __('Task button text is required.'),
                'proof_number.required' => __('Task proof number is required.'),
                'daily_limit.required' => __('Task daily limit is required.'),
            ];

            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'link' => 'required|url',
                'description' => 'required|string|max:1000',
                'category_id' => 'required|integer',
                'reward' => 'required|numeric',
                'limit' => 'required|integer',
                'user_limit' => 'required|integer',
                'country_id' => 'required|integer',
                'city' => 'nullable|integer',
                'gender' => 'required|integer',
                'button_text' => 'required|string|max:500',
                'proof_number' => 'required|integer',
                'daily_limit' => 'required|integer',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            ], $messages);

            // Resim yükleme işlemi
            if ($request->hasFile('image')) {
                // Eski resmi sil
                if ($task->image && file_exists(public_path('assets/images/tasks/' . $task->image))) {
                    unlink(public_path('assets/images/tasks/' . $task->image));
                }
                
                $imageName = time().'.'.$request->image->extension();  
                $request->image->move(public_path('assets/images/tasks'), $imageName);
                $validated['image'] = $imageName;
            }

            $task->update($validated);

            // Cache'i temizle
            Cache::forget('tasks_index_' . Auth::id());

            // AJAX request ise JSON döndür
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => __('Task updated successfully!'),
                    'task' => $task->fresh()
                ]);
            }

            return redirect()->route('tasks.index')->with('success', __('Task updated successfully!'));

        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => __('Validation failed'),
                    'errors' => $e->errors()
                ], 422);
            }
            
            return redirect()->back()->withErrors($e->errors())->withInput();
            
        } catch (\Exception $e) {
            Log::error('Task update error: ' . $e->getMessage());
            
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => __('An error occurred while updating the task')
                ], 500);
            }
            
            return redirect()->back()->with('error', __('An error occurred while updating the task'));
        }
    }

    // Remove the specified resource from storage.
    public function destroy(Task $task)
    {
        $task->delete();
        return redirect()->route('tasks.index');
    }

    // Task listesi sayfası
    public function taskList()
    {
        $tasks = Task::with(['proofs' => function($query) {
            $query->select('TaskId', DB::raw('count(*) as total'))
                  ->where('status', 1)
                  ->groupBy('TaskId');
        }])
        ->leftJoin('socialplatforms', 'tasks.category_id', '=', 'socialplatforms.id')
        ->leftJoin('users', 'tasks.user_id', '=', 'users.id')
        ->leftJoin('cities', 'tasks.city', '=', 'cities.id')
        ->leftJoin('countries', 'tasks.country_id', '=', 'countries.code')
        ->select(
            'tasks.*', 
            'socialplatforms.platform as PlatformName',
            'socialplatforms.icon as PlatformIcon',
            'socialplatforms.description as PlatformDescription',
            'users.account_holder as UserName',
            'users.email as UserEmail',
            'users.user_avatar as UserAvatar',
            'cities.name as CityName',
            'countries.name as CountryName'
        )
        ->where('tasks.status', 1) // Sadece aktif görevler
        ->get();

        $user = Auth::user();
        $platforms = SocialPlatform::where('status', 1)->get();
        $countries = Country::all();
        $cities = City::all();
        
        return view('tasks.list', compact('tasks', 'user', 'platforms', 'countries', 'cities'));
    }

    // Reddedilen kanıtları göster
    public function showDeniedProofs()
    {
        $user = Auth::user();
        return view('taskProofs.denied', compact('user'));
    }

    // Reddedilen kanıtlar için DataTable endpoint
    public function getDeniedProofsDataTable(Request $request)
    {
        $user = Auth::user();
        
        // Base query - sadece kullanıcının kendi reddedilen kanıtları
        $query = \App\Models\TaskProof::with(['task.socialPlatform', 'user', 'files'])
            ->where('UserId', $user->id)
            ->where('Status', 2); // Status 2 = Denied

        // Admin ise tüm reddedilen kanıtları görebilir
        if ($user->user_type == 1) {
            $query = \App\Models\TaskProof::with(['task.socialPlatform', 'user', 'files'])
                ->where('Status', 2);
        }

        // Search functionality
        if ($request->has('search') && $request->search['value']) {
            $searchValue = $request->search['value'];
            $query->whereHas('task', function($q) use ($searchValue) {
                $q->where('name', 'like', "%{$searchValue}%");
            })->orWhereHas('user', function($q) use ($searchValue) {
                $q->where('account_holder', 'like', "%{$searchValue}%");
            })->orWhere('Description', 'like', "%{$searchValue}%")
              ->orWhere('Reason', 'like', "%{$searchValue}%");
        }

        // Total records
        $totalRecords = $query->count();

        // Pagination
        $start = $request->start ?? 0;
        $length = $request->length ?? 10;
        
        $proofs = $query->skip($start)->take($length)->get();

        $data = [];
        foreach ($proofs as $proof) {
            $data[] = [
                'id' => $proof->id,
                'task_name' => $proof->task->name ?? 'N/A',
                'platform' => $proof->task->socialPlatform->platform ?? 'N/A',
                'user_name' => $proof->user->account_holder ?? 'N/A',
                'description' => $proof->Description ?? '',
                'reason' => $proof->Reason ?? __('No reason provided'),
                'files_count' => $proof->files->count(),
                'created_at' => $proof->created_at->format('d.m.Y H:i'),
                'actions' => $this->getDeniedActionButtons($proof, $user)
            ];
        }

        return response()->json([
            'draw' => $request->draw,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $totalRecords,
            'data' => $data
        ]);
    }

    private function getDeniedActionButtons($proof, $user)
    {
        $buttons = '<button class="btn btn-sm btn-info view-denied-proof" data-id="'.$proof->id.'">
                        <i class="bx bx-show"></i> '.__('View').'
                    </button>';
        
        if ($user->user_type == 1) {
            $buttons .= ' <button class="btn btn-sm btn-warning reactivate-proof" data-id="'.$proof->id.'">
                            <i class="bx bx-refresh"></i> '.__('Reactivate').'
                        </button>';
        }
        
        return $buttons;
    }

    // Kanıtı yeniden aktifleştir (admin için)
    public function reactivateProof(Request $request, $id)
    {
        $proof = \App\Models\TaskProof::findOrFail($id);
        $proof->Status = 0; // Pending durumuna geri al
        $proof->Reason = null; // Red sebebini temizle
        $proof->save();
        
        return response()->json([
            'success' => true,
            'message' => __('Proof reactivated successfully.')
        ]);
    }

    // YouTube izleme görevlerine yönlendir
    public function watchedTasks()
    {
        // Clear cache first to avoid old data issues
        Cache::forget('watch_tasks_' . Auth::id());
        
        // Optimize watch tasks query
        $tasks = Cache::remember('watch_tasks_' . Auth::id(), 300, function () {
            return Task::with([
                'proofs' => function($query) {
                    $query->where('status', 1);
                },
                'socialPlatform:id,platform,icon,description',
                'creator:id,account_holder,email,user_avatar',
                'city:id,name',
                'country:code,name'
            ])
            ->where('status', 1)
            ->whereNotNull('watch_time')
            ->where('watch_time', '>', 0)
            ->select('id', 'name', 'link', 'description', 'category_id', 'image', 'reward', 
                    'limit', 'daily_limit', 'user_id', 'city', 'country_id', 'watch_time', 'created_at')
            ->orderBy('created_at', 'desc')
            ->get();
        });

        $user = Auth::user();
        $settings = app('settings'); // Use settings service
        
        return view('tasks.watch-list', compact('tasks', 'user', 'settings'));
    }

    // Task disable metodu (sadece admin)
    public function disableTask(Request $request, $id)
    {
        try {
            $task = Task::findOrFail($id);
            
            // Sadece admin kontrol et
            if (Auth::user()->user_type != 1) {
                return response()->json([
                    'success' => false,
                    'message' => __('Only administrators can disable tasks')
                ], 403);
            }
            
            $task->status = 0; // Pasif yap
            $task->save();
            
            // Cache temizle
            Cache::forget('tasks_index_' . Auth::id());
            
            return response()->json([
                'success' => true,
                'message' => __('Task has been disabled successfully'),
                'new_status' => 0
            ]);
            
        } catch (\Exception $e) {
            Log::error('Task disable error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while disabling the task')
            ], 500);
        }
    }

    // Task confirm metodu
    public function confirmTask(Request $request, $id)
    {
        try {
            $task = Task::findOrFail($id);
            
            // Admin kontrol et
            if (Auth::user()->user_type != 1) {
                return response()->json([
                    'success' => false,
                    'message' => __('You are not authorized to perform this action')
                ], 403);
            }
            
            $task->status = 1; // Onaylandı - Aktif
            $task->save();
            
            // Bildirim gönder
            $notificationService = app(NotificationService::class);
            $notificationService->notifyTaskApproved($task);
            
            // Cache temizle
            Cache::forget('tasks_index_' . Auth::id());
            
            return response()->json([
                'success' => true,
                'message' => __('Task has been approved successfully'),
                'new_status' => 1
            ]);
            
        } catch (\Exception $e) {
            Log::error('Task confirm error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while approving the task')
            ], 500);
        }
    }

    // Task activate metodu
    public function activateTask(Request $request, $id)
    {
        try {
            $task = Task::findOrFail($id);
            
            // Admin veya task sahibi kontrol et
            if (Auth::user()->user_type != 1 && $task->user_id != Auth::id()) {
                return response()->json([
                    'success' => false,
                    'message' => __('You are not authorized to perform this action')
                ], 403);
            }
            
            $task->status = 1; // Aktif yap
            $task->save();
            
            // Cache temizle
            Cache::forget('tasks_index_' . Auth::id());
            
            return response()->json([
                'success' => true,
                'message' => __('Task has been activated successfully'),
                'new_status' => 1
            ]);
            
        } catch (\Exception $e) {
            Log::error('Task activate error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while activating the task')
            ], 500);
        }
    }


    // Task delete metodu (sadece admin)
    public function deleteTask(Request $request, $id)
    {
        try {
            $task = Task::findOrFail($id);
            
            // Sadece admin kontrol et
            if (Auth::user()->user_type != 1) {
                return response()->json([
                    'success' => false,
                    'message' => __('Only administrators can delete tasks')
                ], 403);
            }

            try {
                DB::beginTransaction();

                // Eğer görev reddediliyorsa veya siliniyorsa, görev maliyetini geri iade et
                if ($task->status == '2') { // Onay bekleyen görev
                    $taskCost = $task->reward * $task->limit;
                    $user = User::find($task->user_id);
                    
                    if ($user) {
                        $balanceBefore = $user->balance;
                        $user->increment('balance', $taskCost);
                        $balanceAfter = $user->fresh()->balance;

                        // Balance log kaydı oluştur
                        BalanceLog::create([
                            'user_id' => $user->id,
                            'userId' => $user->id,
                            'TaskId' => $task->id,
                            'amount' => $taskCost,
                            'balance' => $taskCost,
                            'balance_before' => $balanceBefore,
                            'balance_after' => $balanceAfter,
                            'operation_type' => 'credit',
                            'type' => 'task_refund',
                            'description' => __('Refund for deleted task ":task_name"', ['task_name' => $task->name]),
                            'status' => 'completed',
                            'reference_type' => 'task',
                            'reference_id' => $task->id,
                            'added_user' => Auth::id()
                        ]);

                        // Transaction log kaydı oluştur
                        TransactionLog::create([
                            'user_id' => $user->id,
                            'type' => 'task_refund',
                            'amount' => $taskCost,
                            'description' => __('Refund for deleted task ":task_name"', ['task_name' => $task->name]),
                            'operation_type' => 'credit',
                            'reference_type' => 'task',
                            'reference_id' => $task->id,
                            'balance_before' => $balanceBefore,
                            'balance_after' => $balanceAfter
                        ]);
                    }
                }
                
                // Resmi sil
                if ($task->image && file_exists(public_path('assets/images/tasks/' . $task->image))) {
                    unlink(public_path('assets/images/tasks/' . $task->image));
                }
                
                $task->delete();

                DB::commit();
                
                // Cache temizle
                Cache::forget('tasks_index_' . Auth::id());
                
                return response()->json([
                    'success' => true,
                    'message' => __('Task has been deleted successfully')
                ]);

            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }
            
        } catch (\Exception $e) {
            Log::error('Task delete error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while deleting the task')
            ], 500);
        }
    }

    // Silinen/Aktif olmayan görevleri listele
    public function unvisibleTasks()
    {
        $user = Auth::user();
        
        // Sosyal medya görevleri (status = '0' veya visibilty = 0)
        $socialTasks = Task::with(['socialPlatform:id,platform,icon', 'creator:id,account_holder'])
            ->where(function($query) {
                $query->where('status', '0')
                      ->orWhere('visibilty', 0);
            })
            ->whereNotNull('category_id')
            ->orderBy('updated_at', 'desc')
            ->get();

        // Website ziyaret görevleri (status = 0)
        $websiteTasks = WebsiteVisitTask::with(['creator:id,account_holder', 'country:id,name', 'city:id,name'])
            ->where('status', 0)
            ->orderBy('updated_at', 'desc')
            ->get();

        // YouTube izleme görevleri (is_active = 0)
        $watchTasks = WatchTask::with(['creator:id,account_holder'])
            ->where('is_active', 0)
            ->orderBy('updated_at', 'desc')
            ->get();

        return view('tasks.deleted', compact('user', 'socialTasks', 'websiteTasks', 'watchTasks'));
    }
}
