<?php

namespace App\Http\Controllers;

use App\Models\Task;
use App\Models\TaskProof;
use App\Models\TaskProofFile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class TaskProofController extends Controller
{
    
        
        
    public function index()
    {
        $user = Auth::user();
        return view('taskProofs.index', compact('user'));
    }

    // DataTable için AJAX endpoint
    public function getProofsDataTable(Request $request)
    {
        $user = Auth::user();
        
        // Base query - sadece kullanıcının kendi kanıtları
        $query = TaskProof::with(['task.socialPlatform', 'user', 'files'])
            ->where('UserId', $user->id)
            ->where('Status', 0);

        // Admin ise tüm kanıtları görebilir
        if ($user->user_type == 1) {
            $query = TaskProof::with(['task.socialPlatform', 'user', 'files'])
                ->where('Status', 0);
        }

        // Search functionality
        if ($request->has('search') && $request->search['value']) {
            $searchValue = $request->search['value'];
            $query->whereHas('task', function($q) use ($searchValue) {
                $q->where('name', 'like', "%{$searchValue}%");
            })->orWhereHas('user', function($q) use ($searchValue) {
                $q->where('account_holder', 'like', "%{$searchValue}%");
            })->orWhere('Description', 'like', "%{$searchValue}%");
        }

        // Total records
        $totalRecords = $query->count();

        // Pagination
        $start = $request->start ?? 0;
        $length = $request->length ?? 10;
        
        $proofs = $query->skip($start)->take($length)->get();

        $data = [];
        foreach ($proofs as $proof) {
            $data[] = [
                'id' => $proof->id,
                'task_name' => $proof->task->name ?? 'N/A',
                'platform' => $proof->task->socialPlatform->platform ?? 'N/A',
                'user_name' => $proof->user->account_holder ?? 'N/A',
                'description' => $proof->Description ?? '',
                'files_count' => $proof->files->count(),
                'created_at' => $proof->created_at->format('d.m.Y H:i'),
                'actions' => $this->getActionButtons($proof, $user)
            ];
        }

        return response()->json([
            'draw' => $request->draw,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $totalRecords,
            'data' => $data
        ]);
    }

    private function getActionButtons($proof, $user)
    {
        $buttons = '<button class="btn btn-sm btn-info view-proof" data-id="'.$proof->id.'">
                        <i class="bx bx-show"></i> '.__('View').'
                    </button>';
        
        if ($user->user_type == 1) {
            $buttons .= ' <button class="btn btn-sm btn-success approve-proof" data-id="'.$proof->id.'">
                            <i class="bx bx-check"></i> '.__('Approve').'
                        </button>
                        <button class="btn btn-sm btn-danger deny-proof" data-id="'.$proof->id.'">
                            <i class="bx bx-x"></i> '.__('Deny').'
                        </button>';
        }
        
        return $buttons;
    }
    

    // Show the form for creating a new resource
    public function create()
    {
        return view('taskProofs.create');
    }

    
    public function store(Request $request)
    {
        try {
            $task = Task::findOrFail($request->id);
            
            // Kullanıcının bu görev için bekleyen proof'u var mı kontrol et
            $existingProof = TaskProof::where('TaskId', $task->id)
                ->where('UserId', Auth::id())
                ->where('Status', 0)
                ->first();
                
            if ($existingProof) {
                return response()->json([
                    'success' => false,
                    'message' => __('You already have a pending proof for this task.')
                ]);
            }
    
            if (!$request->hasFile('proofs')) {
                return response()->json([
                    'success' => false,
                    'message' => __('Please upload at least one proof file.')
                ]);
            }
          
            $taskProof = TaskProof::create([
                'UserId' => Auth::id(),
                'TaskId' => $task->id,
                'Description' => $request->input('Description'),
                'Status' => 0,
                'created_at' => now(),
                'updated_at' => now()
            ]);
    
            foreach ($request->file('proofs') as $proof) {
                $extension = $proof->getClientOriginalExtension();
                $fileName = uniqid() . '.' . $extension;
                $proof->move(public_path('assets/images/taskProofs'), $fileName);
    
                TaskProofFile::create([
                    'proofId' => $taskProof->id,
                    'proofImage' => $fileName,
                    'status' => 0,
                    'created_at' => now(),
                    'updated_at' => now()
                ]);
            }
    
            $task->increment('count_proof');
            
            return response()->json([
                'success' => true,
                'message' => __('Your proofs have been submitted successfully. Please wait for approval.')
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while submitting your proof. Please try again.')
            ]);
        }
    }
    
    

    
public function show($id)
{
    $proof = TaskProof::with('user', 'task', 'images')->find($id);

    if (!$proof) {
        return response()->json([
            'success' => false,
            'message' => 'Proof not found.'
        ]);
    }

    $detailsHtml = view('taskProofs.proof_details', compact('proof'))->render();
    $imagesHtml = view('taskProofs.proof_images', compact('proof'))->render();
    $userHtml = view('taskProofs.proof_user', compact('proof'))->render();
    
    return response()->json([
        'success' => true,
        'detailsHtml' => $detailsHtml,
        'imagesHtml' => $imagesHtml,
        'userHtml' => $userHtml,
    ]);
}
    
    public function updateStatus(Request $request, $id)
    {
        $proof = TaskProof::findOrFail($id);
        $proof->status = $request->status;
        $proof->save();
    
        return response()->json(['success' => true, 'message' => __('Proof status updated successfully.')]);
    }
    
    // Proof onaylama
    public function approveProof(Request $request, $id)
    {
        $proof = TaskProof::findOrFail($id);
        $proof->Status = 1;
        $proof->save();
        
        return response()->json([
            'success' => true,
            'message' => __('Proof approved successfully.')
        ]);
    }
    
    // Proof reddetme
    public function denyProof(Request $request, $id)
    {
        $proof = TaskProof::findOrFail($id);
        $proof->Status = 2;
        $proof->Reason = $request->reason ?? __('Proof denied by admin.');
        $proof->save();
        
        return response()->json([
            'success' => true,
            'message' => __('Proof denied successfully.')
        ]);
    }
    

    public function edit(TaskProof $taskProof)
    {
        return view('taskProofs.edit', compact('taskProof'));
    }

    public function update(Request $request, TaskProof $taskProof)
    {
        $validatedData = $request->validate([
            'UserId' => 'required|integer',
            'TaskId' => 'required|integer',
            'ProofImg' => 'required|string|max:255',
            'Description' => 'nullable|text',
            'Status' => 'required|integer',
            'Reason' => 'nullable|text',
        ]);
        $taskProof->update($validatedData);
        return redirect('/taskProofs')->with('success', __('Task Proof has been updated'));
    }

    public function destroy(TaskProof $taskProof)
    {
        $taskProof->delete();
        return redirect('/taskProofs')->with('success', __('Task Proof has been deleted'));
    }
}
