<?php

namespace App\Http\Controllers;

use App\Models\Ticket;
use App\Models\TicketMessage;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class TicketController extends Controller
{
    // User: Ticket listesi
    public function index(Request $request)
    {
        $user = Auth::user();
        
        if ($user->user_type == 1) {
            // Admin view
            return $this->adminIndex($request);
        }
        
        // User view
        $query = Ticket::where('user_id', $user->id)
            ->with(['latestMessage', 'assignedTo']);
            
        // Filters
        if ($request->status) {
            if ($request->status === 'open') {
                $query->open();
            } elseif ($request->status === 'closed') {
                $query->closed();
            } else {
                $query->where('status', $request->status);
            }
        }
        
        if ($request->category) {
            $query->where('category', $request->category);
        }
        
        $tickets = $query->orderBy('last_activity_at', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return view('tickets.index', compact('tickets'));
    }
    
    // Admin: Ticket listesi
    public function adminIndex(Request $request)
    {
        $query = Ticket::with(['user', 'latestMessage', 'assignedTo']);
        
        // Filters
        if ($request->status) {
            if ($request->status === 'open') {
                $query->open();
            } elseif ($request->status === 'closed') {
                $query->closed();
            } else {
                $query->where('status', $request->status);
            }
        }
        
        if ($request->priority) {
            $query->where('priority', $request->priority);
        }
        
        if ($request->category) {
            $query->where('category', $request->category);
        }
        
        if ($request->assigned) {
            if ($request->assigned === 'me') {
                $query->where('assigned_to', Auth::id());
            } elseif ($request->assigned === 'unassigned') {
                $query->whereNull('assigned_to');
            }
        }
        
        $tickets = $query->orderBy('priority', 'desc')
            ->orderBy('last_activity_at', 'desc')
            ->paginate(20);
            
        // Statistics
        $stats = [
            'total' => Ticket::count(),
            'open' => Ticket::open()->count(),
            'closed' => Ticket::closed()->count(),
            'urgent' => Ticket::where('priority', 'urgent')->open()->count(),
            'assigned_to_me' => Ticket::where('assigned_to', Auth::id())->open()->count(),
            'unassigned' => Ticket::whereNull('assigned_to')->open()->count()
        ];
            
        return view('tickets.admin.index', compact('tickets', 'stats'));
    }
    
    // Ticket oluşturma formu
    public function create()
    {
        return view('tickets.create');
    }
    
    // Ticket oluştur
    public function store(Request $request)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'description' => 'required|string|max:5000',
            'category' => 'required|in:general,technical,billing,account,withdrawal,task',
            'priority' => 'required|in:low,medium,high,urgent',
            'attachments.*' => 'file|max:10240|mimes:jpg,jpeg,png,pdf,doc,docx,txt'
        ]);
        
        DB::beginTransaction();
        try {
            $user = Auth::user();
            
            // Handle file uploads
            $attachments = [];
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $path = $file->store('ticket-attachments', 'public');
                    $attachments[] = [
                        'path' => $path,
                        'name' => $file->getClientOriginalName(),
                        'size' => $file->getSize(),
                        'type' => $file->getMimeType()
                    ];
                }
            }
            
            // Create ticket
            $ticket = Ticket::create([
                'ticket_number' => Ticket::generateTicketNumber(),
                'user_id' => $user->id,
                'subject' => $request->subject,
                'description' => $request->description,
                'category' => $request->category,
                'priority' => $request->priority,
                'status' => 'open',
                'last_activity_at' => now(),
                'attachments' => $attachments
            ]);
            
            // Create initial message
            TicketMessage::create([
                'ticket_id' => $ticket->id,
                'user_id' => $user->id,
                'message' => $request->description,
                'attachments' => $attachments
            ]);
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => __('Ticket created successfully'),
                'ticket_id' => $ticket->id,
                'ticket_number' => $ticket->ticket_number
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('Error creating ticket'),
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    // Ticket detayı
    public function show($id)
    {
        $user = Auth::user();
        
        $ticket = Ticket::with(['user', 'assignedTo', 'messages.user'])
            ->findOrFail($id);
            
        // Check permissions
        if ($user->user_type != 1 && $ticket->user_id != $user->id) {
            abort(403);
        }
        
        // Mark messages as read
        $ticket->messages()
            ->where('user_id', '!=', $user->id)
            ->where('is_read', false)
            ->update(['is_read' => true, 'read_at' => now()]);
            
        return view('tickets.show', compact('ticket'));
    }
    
    // Mesaj gönder
    public function sendMessage(Request $request, $id)
    {
        try {
            $request->validate([
                'message' => 'required|string|max:5000',
                'attachments.*' => 'file|max:10240|mimes:jpg,jpeg,png,pdf,doc,docx,txt',
                'is_internal' => 'boolean'
            ]);
            
            $user = Auth::user();
            $ticket = Ticket::findOrFail($id);
        
        // Check permissions
        if ($user->user_type != 1 && $ticket->user_id != $user->id) {
            abort(403);
        }
        
        // Handle file uploads
        $attachments = [];
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('ticket-attachments', 'public');
                $attachments[] = [
                    'path' => $path,
                    'name' => $file->getClientOriginalName(),
                    'size' => $file->getSize(),
                    'type' => $file->getMimeType()
                ];
            }
        }
        
        // Create message
        $message = TicketMessage::create([
            'ticket_id' => $ticket->id,
            'user_id' => $user->id,
            'message' => $request->message,
            'attachments' => $attachments,
            'is_internal' => $request->is_internal && $user->user_type == 1
        ]);
        
        // Update ticket status and activity
        $newStatus = $ticket->status;
        if ($user->user_type == 1) {
            // Admin reply
            if ($ticket->status == 'waiting_admin') {
                $newStatus = 'waiting_user';
            }
        } else {
            // User reply
            if ($ticket->status == 'waiting_user') {
                $newStatus = 'waiting_admin';
            } elseif ($ticket->status == 'resolved') {
                $newStatus = 'open';
            }
        }
        
        $ticket->update([
            'status' => $newStatus,
            'last_activity_at' => now()
        ]);
        
        return response()->json([
            'success' => true,
            'message' => __('Message sent successfully'),
            'message_data' => [
                'id' => $message->id,
                'user_name' => $user->account_holder ?? $user->name,
                'user_type' => $user->user_type,
                'message' => $message->formatted_message,
                'created_at' => $message->created_at->format('d.m.Y H:i'),
                'attachments' => $message->attachments ?? [],
                'is_internal' => $message->is_internal ?? false
            ]
        ]);
        
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }
    
    // Ticket durumu güncelle (Admin)
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:open,in_progress,waiting_user,waiting_admin,resolved,closed',
            'note' => 'nullable|string|max:1000'
        ]);
        
        $user = Auth::user();
        if ($user->user_type != 1) {
            abort(403);
        }
        
        $ticket = Ticket::findOrFail($id);
        
        $oldStatus = $ticket->status;
        $ticket->update([
            'status' => $request->status,
            'last_activity_at' => now(),
            'resolved_at' => $request->status == 'resolved' ? now() : null,
            'closed_at' => $request->status == 'closed' ? now() : null
        ]);
        
        // Add internal note if provided
        if ($request->note) {
            TicketMessage::create([
                'ticket_id' => $ticket->id,
                'user_id' => $user->id,
                'message' => "Status changed from {$oldStatus} to {$request->status}.\n\nNote: {$request->note}",
                'is_internal' => true
            ]);
        }
        
        return response()->json([
            'success' => true,
            'message' => __('Ticket status updated successfully')
        ]);
    }
    
    // Admin ticket oluştur
    public function adminStore(Request $request)
    {
        try {
            $request->validate([
                'user_id' => 'required|exists:users,id',
                'subject' => 'required|string|max:255',
                'description' => 'required|string|max:5000',
                'category' => 'required|in:general,technical,billing,account,withdrawal,task',
                'priority' => 'required|in:low,medium,high,urgent',
                'attachments.*' => 'file|max:10240|mimes:jpg,jpeg,png,pdf,doc,docx,txt',
                'assign_to_me' => 'boolean'
            ]);
            
            $admin = Auth::user();
            if ($admin->user_type != 1) {
                abort(403);
            }
            
            DB::beginTransaction();
            
            // Handle file uploads
            $attachments = [];
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $path = $file->store('ticket-attachments', 'public');
                    $attachments[] = [
                        'path' => $path,
                        'name' => $file->getClientOriginalName(),
                        'size' => $file->getSize(),
                        'type' => $file->getMimeType()
                    ];
                }
            }
            
            // Create ticket
            $ticket = Ticket::create([
                'ticket_number' => Ticket::generateTicketNumber(),
                'user_id' => $request->user_id,
                'subject' => $request->subject,
                'description' => $request->description,
                'category' => $request->category,
                'priority' => $request->priority,
                'status' => 'open',
                'assigned_to' => $request->assign_to_me ? $admin->id : null,
                'last_activity_at' => now(),
                'attachments' => $attachments
            ]);
            
            // Create initial message
            TicketMessage::create([
                'ticket_id' => $ticket->id,
                'user_id' => $admin->id,
                'message' => $request->description,
                'attachments' => $attachments,
                'is_internal' => false
            ]);
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => __('Ticket created successfully'),
                'ticket_id' => $ticket->id,
                'ticket_number' => $ticket->ticket_number
            ]);
            
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('Error creating ticket')
            ], 500);
        }
    }

    // Ticket atama (Admin)
    public function assign(Request $request, $id)
    {
        $request->validate([
            'assigned_to' => 'nullable|exists:users,id'
        ]);
        
        $user = Auth::user();
        if ($user->user_type != 1) {
            abort(403);
        }
        
        $ticket = Ticket::findOrFail($id);
        $ticket->update([
            'assigned_to' => $request->assigned_to,
            'last_activity_at' => now()
        ]);
        
        return response()->json([
            'success' => true,
            'message' => __('Ticket assigned successfully')
        ]);
    }
    
    // DataTable API (Admin)
    public function adminDataTable(Request $request)
    {
        try {
            $query = Ticket::with(['user', 'latestMessage', 'assignedTo']);
        
        // Search
        if ($request->has('search') && $request->search['value']) {
            $searchValue = $request->search['value'];
            $query->where(function($q) use ($searchValue) {
                $q->where('ticket_number', 'like', "%{$searchValue}%")
                  ->orWhere('subject', 'like', "%{$searchValue}%")
                  ->orWhereHas('user', function($userQuery) use ($searchValue) {
                      $userQuery->where('account_holder', 'like', "%{$searchValue}%")
                               ->orWhere('email', 'like', "%{$searchValue}%");
                  });
            });
        }
        
        // Filters
        if ($request->status) {
            if ($request->status === 'open') {
                $query->open();
            } elseif ($request->status === 'closed') {
                $query->closed();
            } else {
                $query->where('status', $request->status);
            }
        }
        
        if ($request->priority) {
            $query->where('priority', $request->priority);
        }
        
        if ($request->category) {
            $query->where('category', $request->category);
        }
        
        $totalRecords = $query->count();
        
        // Pagination
        $start = $request->start ?? 0;
        $length = $request->length ?? 25;
        
        $tickets = $query->skip($start)->take($length)
            ->orderBy('priority', 'desc')
            ->orderBy('last_activity_at', 'desc')
            ->get();
            
        $data = [];
        foreach ($tickets as $ticket) {
            $data[] = [
                'id' => $ticket->id,
                'ticket_number' => $ticket->ticket_number,
                'user_name' => $ticket->user->account_holder ?? $ticket->user->name,
                'user_email' => $ticket->user->email,
                'subject' => $ticket->subject,
                'category' => $ticket->category,
                'category_text' => $ticket->category_text,
                'priority' => $ticket->priority,
                'priority_text' => $ticket->priority_text,
                'priority_badge' => $ticket->priority_badge,
                'status' => $ticket->status,
                'status_text' => $ticket->status_text,
                'status_badge' => $ticket->status_badge,
                'assigned_to' => $ticket->assignedTo->account_holder ?? $ticket->assignedTo->name ?? null,
                'last_activity_at' => $ticket->last_activity_at->format('d.m.Y H:i'),
                'created_at' => $ticket->created_at->format('d.m.Y H:i'),
                'unread_count' => $ticket->getUnreadCountForUser(Auth::id())
            ];
        }
        
        $response = [
            'draw' => $request->draw,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $totalRecords,
            'data' => $data
        ];

        return response()->json($response);
        
        } catch (\Exception $e) {
            return response()->json([
                'draw' => $request->draw ?? 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading data'
            ], 500);
        }
    }
    
    // Mobile API
    public function mobileData(Request $request)
    {
        $user = Auth::user();
        $query = Ticket::where('user_id', $user->id)
            ->with(['latestMessage', 'assignedTo']);
            
        // Filters
        if ($request->status) {
            if ($request->status === 'open') {
                $query->open();
            } elseif ($request->status === 'closed') {
                $query->closed();
            } else {
                $query->where('status', $request->status);
            }
        }
        
        $start = $request->start ?? 0;
        $length = $request->length ?? 10;
        
        $tickets = $query->skip($start)->take($length)
            ->orderBy('last_activity_at', 'desc')
            ->get();
            
        $data = [];
        foreach ($tickets as $ticket) {
            $data[] = [
                'id' => $ticket->id,
                'ticket_number' => $ticket->ticket_number,
                'subject' => $ticket->subject,
                'category' => $ticket->category,
                'category_text' => $ticket->category_text,
                'priority' => $ticket->priority,
                'priority_text' => $ticket->priority_text,
                'priority_badge' => $ticket->priority_badge,
                'status' => $ticket->status,
                'status_text' => $ticket->status_text,
                'status_badge' => $ticket->status_badge,
                'last_activity_at' => $ticket->last_activity_at->diffForHumans(),
                'created_at' => $ticket->created_at->format('d.m.Y'),
                'unread_count' => $ticket->getUnreadCountForUser($user->id),
                'latest_message' => $ticket->latestMessage ? [
                    'message' => Str::limit($ticket->latestMessage->message, 100),
                    'user_name' => $ticket->latestMessage->user->account_holder ?? $ticket->latestMessage->user->name,
                    'is_admin' => $ticket->latestMessage->is_admin,
                    'created_at' => $ticket->latestMessage->created_at->diffForHumans()
                ] : null
            ];
        }
        
        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }
}