<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\WebsiteVisitTask;
use App\Models\Country;
use App\Models\City;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WebsiteVisitTaskController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display user's own website visit tasks
     */
    public function index()
    {
        $user = Auth::user();
        
        $tasks = WebsiteVisitTask::where('created_by', $user->id)
            ->with(['country', 'city'])
            ->withCount(['visits', 'completedVisits'])
            ->orderBy('created_at', 'desc')
            ->get(); // Remove pagination for AJAX

        // Return JSON for AJAX requests
        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $tasks
            ]);
        }

        return view('user.website-visits.index', compact('tasks'));
    }

    /**
     * Show the form for creating a new website visit task
     */
    public function create()
    {
        $countries = Country::all();
        $cities = City::all();
        
        // Return JSON for AJAX requests
        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'countries' => $countries,
                'cities' => $cities
            ]);
        }
        
        return view('user.website-visits.create', compact('countries', 'cities'));
    }

    /**
     * Store a newly created website visit task
     */
    public function store(Request $request)
    {
        $messages = [
            'title.required' => __('Task title is required.'),
            'website_url.required' => __('Website URL is required.'),
            'website_url.url' => __('Please enter a valid URL.'),
            'required_time.required' => __('Required time is required.'),
            'required_time.min' => __('Required time must be at least 10 seconds.'),
            'reward_amount.required' => __('Reward amount is required.'),
            'reward_amount.min' => __('Reward amount must be greater than 0.'),
            'max_completions.required' => __('Maximum completions is required.'),
            'user_daily_limit.required' => __('User daily limit is required.'),
            'user_total_limit.required' => __('User total limit is required.'),
        ];

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'website_url' => 'required|url',
            'visit_method' => 'required|in:direct,google_search',
            'search_keywords' => 'required_if:visit_method,google_search|string|max:255',
            'required_time' => 'required|integer|min:10|max:3600',
            'reward_amount' => 'required|numeric|min:0.01',
            'max_completions' => 'required|integer|min:1',
            'user_daily_limit' => 'required|integer|min:1|max:50',
            'user_total_limit' => 'required|integer|min:1',
            'target_country_id' => 'nullable|integer',
            'target_city_id' => 'nullable|integer',
            'target_gender' => 'required|in:0,1,2',
            'expires_at' => 'nullable|date|after:now',
        ], $messages);

        $validated['created_by'] = Auth::id();
        $validated['target_country_id'] = $request->input('target_country_id', 0);
        $validated['target_city_id'] = $request->input('target_city_id', 0);
        
        // Convert empty strings to 0
        if (empty($validated['target_country_id']) || $validated['target_country_id'] === '') {
            $validated['target_country_id'] = 0;
        }
        if (empty($validated['target_city_id']) || $validated['target_city_id'] === '') {
            $validated['target_city_id'] = 0;
        }

        $task = WebsiteVisitTask::create($validated);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => __('Website visit task created successfully.'),
                'task' => $task
            ]);
        }

        // Return JSON for AJAX requests
        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => __('Website visit task created successfully.'),
                'data' => $task
            ]);
        }

        return redirect()->route('user.website-visits.index')
            ->with('success', __('Website visit task created successfully.'));
    }

    /**
     * Show the form for editing the specified task
     */
    public function edit(WebsiteVisitTask $task)
    {
        $this->authorize('update', $task);
        
        $countries = Country::all();
        $cities = City::all();
        
        return view('user.website-visits.edit', compact('task', 'countries', 'cities'));
    }

    /**
     * Update the specified task
     */
    public function update(Request $request, WebsiteVisitTask $task)
    {
        $this->authorize('update', $task);
        
        $messages = [
            'title.required' => __('Task title is required.'),
            'website_url.required' => __('Website URL is required.'),
            'website_url.url' => __('Please enter a valid URL.'),
            'required_time.required' => __('Required time is required.'),
            'required_time.min' => __('Required time must be at least 10 seconds.'),
            'reward_amount.required' => __('Reward amount is required.'),
            'reward_amount.min' => __('Reward amount must be greater than 0.'),
            'max_completions.required' => __('Maximum completions is required.'),
            'user_daily_limit.required' => __('User daily limit is required.'),
            'user_total_limit.required' => __('User total limit is required.'),
        ];

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'website_url' => 'required|url',
            'visit_method' => 'required|in:direct,google_search',
            'search_keywords' => 'required_if:visit_method,google_search|string|max:255',
            'required_time' => 'required|integer|min:10|max:3600',
            'reward_amount' => 'required|numeric|min:0.01',
            'max_completions' => 'required|integer|min:1',
            'user_daily_limit' => 'required|integer|min:1|max:50',
            'user_total_limit' => 'required|integer|min:1',
            'target_country_id' => 'nullable|integer',
            'target_city_id' => 'nullable|integer',
            'target_gender' => 'required|in:0,1,2',
            'expires_at' => 'nullable|date|after:now',
        ], $messages);

        $validated['target_country_id'] = $request->input('target_country_id', 0);
        $validated['target_city_id'] = $request->input('target_city_id', 0);
        
        // Convert empty strings to 0
        if (empty($validated['target_country_id']) || $validated['target_country_id'] === '') {
            $validated['target_country_id'] = 0;
        }
        if (empty($validated['target_city_id']) || $validated['target_city_id'] === '') {
            $validated['target_city_id'] = 0;
        }

        $task->update($validated);
 
        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => __('Website visit task updated successfully.'),
                'task' => $task
            ]);
        }

        // Return JSON for AJAX requests
        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => __('Website visit task updated successfully.'),
                'data' => $task
            ]);
        }

        return redirect()->route('user.website-visits.index')
            ->with('success', __('Website visit task updated successfully.'));
    }

    /**
     * Show task details and statistics
     */
    public function show(WebsiteVisitTask $task)
    {
        $this->authorize('view', $task);
        
        $task->load(['country', 'city']);
        
        // Return JSON for AJAX requests
        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $task
            ]);
        }
        
        $stats = [
            'total_visits' => $task->visits()->count(),
            'completed_visits' => $task->completedVisits()->count(),
            'failed_visits' => $task->visits()->where('status', 'failed')->count(),
            'unique_users' => $task->visits()->distinct('user_id')->count(),
            'avg_duration' => $task->completedVisits()->avg('duration'),
            'today_visits' => $task->visits()->whereDate('created_at', today())->count(),
        ];

        $recentVisits = $task->visits()
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // For regular page view, use $websiteVisit to match the view
        $websiteVisit = $task;

        return view('user.website-visits.show', compact('websiteVisit', 'stats', 'recentVisits'));
    }

    /**
     * Remove the specified task from storage
     */
    public function destroy(WebsiteVisitTask $task)
    {
        $this->authorize('update', $task);
        
        try {
            // Delete related visits and proofs first
            $task->visits()->delete();
            
            // Delete the task
            $task->delete();
            
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => __('Website visit task deleted successfully.')
                ]);
            }
            
            return redirect()->route('user.website-visits.index')
                ->with('success', __('Website visit task deleted successfully.'));
                
        } catch (\Exception $e) {
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => __('An error occurred while deleting the task.')
                ], 500);
            }
            
            return redirect()->route('user.website-visits.index')
                ->with('error', __('An error occurred while deleting the task.'));
        }
    }
}