<?php
namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use App\Models\BalanceLog;
use App\Models\Notification;
use App\Models\TaskProofFile;
use App\Models\TaskUser;
use App\Models\UserSession;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Auth;
use App\Models\SocialPlatform;
use Illuminate\Support\Facades\Cache;
use App\Models\Task;
use Carbon\Carbon;
use App\Models\Country;
use App\Models\City;
use App\Models\TaskProof;

class UserController extends Controller
{
    public function listUsersWithSameIp(Request $request)
    {
        // Aynı IP adresine sahip olan kullanıcıları gruplandırma ve bu IP'ye sahip tüm kullanıcıları listeleme
        // Localhost ve test IP'lerini hariç tut
        $usersGroupedByIp = User::select('ip')
            ->whereNotNull('ip')
            ->where('ip', '!=', '')
            ->where('ip', '!=', 'localhost')
            ->where('ip', '!=', '127.0.0.1')
            ->where('ip', '!=', '::1')
            ->where('ip', 'NOT LIKE', '192.168.%')
            ->where('ip', 'NOT LIKE', '10.%')
            ->where('ip', 'NOT LIKE', '172.16.%')
            ->where('ip', 'NOT LIKE', '172.17.%')
            ->where('ip', 'NOT LIKE', '172.18.%')
            ->where('ip', 'NOT LIKE', '172.19.%')
            ->where('ip', 'NOT LIKE', '172.2_.%')
            ->where('ip', 'NOT LIKE', '172.30.%')
            ->where('ip', 'NOT LIKE', '172.31.%')
            ->groupBy('ip')
            ->havingRaw('COUNT(*) > 1')
            ->pluck('ip');

        // Bu IP adreslerine sahip olan tüm kullanıcıları alalım
        $users = User::whereIn('ip', $usersGroupedByIp)->get();

        $user = Auth::user();

        $search = $request->input('search');

        $startDate = Carbon::now()->startOfWeek();
        $endDate = Carbon::now()->endOfWeek();
        $platforms = SocialPlatform::where('status', 1)->get();
        $mobileverifed = User::where('phone_verified', 1)->count();
        $emailverifed = User::where('verified', 1)->count();
        $totaluser = User::where('membership_type', 1)->count();
        $totalBalance = User::sum('balance');
        $todayRegistrations = User::whereDate('created_at', today())->count();
        $tasksAwaitingApproval = Task::where('status', 0)->count();
        $activeTasks = Task::where('status', 1)->count();
        $completedTasks = Task::where('status', 2)->count();
        $weeklyTask = Task::whereBetween('created_at', [$startDate, $endDate])->count();
        $onlineUsersCount = UserSession::where('is_active', true)->count();
        $evidencesAwaitingApproval = TaskProof::where('status', 0)->count();

        // Sorgu oluşturma
        $query = User::select(
            'users.*',
            'cities.name as CityName',
            'countries.name as CountryName'
        )
        ->leftJoin('cities', 'cities.id', '=', 'users.city_id')
        ->leftJoin('countries', 'countries.code', '=', 'users.country_id')
        ->where('users.user_type', '!=', 1);

        // Arama filtresi uygulama
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('users.account_holder', 'like', "%{$search}%")
                  ->orWhere('users.username', 'like', "%{$search}%")
                  ->orWhere('users.email', 'like', "%{$search}%")
                  ->orWhere('users.phone', 'like', "%{$search}%")
                  ->orWhere('cities.name', 'like', "%{$search}%")
                  ->orWhere('countries.name', 'like', "%{$search}%");
            });
        }

        // Sayfalama işlemi
        $members = $query->paginate(10);

        return view('users.list_same_ip', compact('members','users','platforms', 'user', 'onlineUsersCount', 'mobileverifed', 'emailverifed', 'totaluser', 'totalBalance', 'todayRegistrations', 'tasksAwaitingApproval', 'activeTasks', 'completedTasks', 'weeklyTask','evidencesAwaitingApproval'));
    }

    public function getSameIpUsersAjax(Request $request)
    {
        $search = $request->input('search');
        $page = $request->input('page', 1);
        $perPage = $request->input('per_page', 12);

        // Aynı IP adresine sahip olan kullanıcıları gruplandırma
        // Localhost ve private IP'leri hariç tut
        $usersGroupedByIp = User::select('ip')
            ->whereNotNull('ip')
            ->where('ip', '!=', '')
            ->where('ip', '!=', 'localhost')
            ->where('ip', '!=', '127.0.0.1')
            ->where('ip', '!=', '::1')
            ->where('ip', 'NOT LIKE', '192.168.%')
            ->where('ip', 'NOT LIKE', '10.%')
            ->where('ip', 'NOT LIKE', '172.16.%')
            ->where('ip', 'NOT LIKE', '172.17.%')
            ->where('ip', 'NOT LIKE', '172.18.%')
            ->where('ip', 'NOT LIKE', '172.19.%')
            ->where('ip', 'NOT LIKE', '172.2_.%')
            ->where('ip', 'NOT LIKE', '172.30.%')
            ->where('ip', 'NOT LIKE', '172.31.%')
            ->groupBy('ip')
            ->havingRaw('COUNT(*) > 1')
            ->pluck('ip');

        // Bu IP adreslerine sahip olan tüm kullanıcıları alalım
        $query = User::select(
            'users.*',
            'cities.name as CityName',
            'countries.name as CountryName'
        )
        ->leftJoin('cities', 'cities.id', '=', 'users.city_id')
        ->leftJoin('countries', 'countries.code', '=', 'users.country_id')
        ->whereIn('users.ip', $usersGroupedByIp)
        ->where('users.user_type', '!=', 1);

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('users.account_holder', 'like', "%{$search}%")
                  ->orWhere('users.username', 'like', "%{$search}%")
                  ->orWhere('users.email', 'like', "%{$search}%")
                  ->orWhere('users.phone', 'like', "%{$search}%")
                  ->orWhere('users.ip', 'like', "%{$search}%")
                  ->orWhere('cities.name', 'like', "%{$search}%")
                  ->orWhere('countries.name', 'like', "%{$search}%");
            });
        }

        $members = $query->orderBy('users.ip')->paginate($perPage, ['*'], 'page', $page);

        // IP gruplarını oluştur
        $ipGroups = [];
        foreach ($members->items() as $member) {
            $ipGroups[$member->ip][] = $member;
        }

        return response()->json([
            'success' => true,
            'data' => $members->items(),
            'ip_groups' => $ipGroups,
            'pagination' => [
                'current_page' => $members->currentPage(),
                'last_page' => $members->lastPage(),
                'per_page' => $members->perPage(),
                'total' => $members->total(),
                'from' => $members->firstItem(),
                'to' => $members->lastItem()
            ]
        ]);
    }

    public function getSameIpUserDetails($id)
    {
        $user = User::select(
            'users.*',
            'cities.name as CityName',
            'countries.name as CountryName'
        )
        ->leftJoin('cities', 'cities.id', '=', 'users.city_id')
        ->leftJoin('countries', 'countries.code', '=', 'users.country_id')
        ->find($id);

        if ($user) {
            // Aynı IP'ye sahip diğer kullanıcıları da getir
            $sameIpUsers = User::select('id', 'username', 'account_holder', 'email', 'created_at')
                ->where('ip', $user->ip)
                ->where('id', '!=', $user->id)
                ->get();

            return response()->json([
                'success' => true, 
                'user' => $user,
                'same_ip_users' => $sameIpUsers
            ]);
        }
        return response()->json(['success' => false]);
    }

    public function deleteMember($id, Request $request)
    {
        $user = User::find($id);
        if ($user) {
            $user->deleted = 1;
            $user->deletion_reason = $request->reason;
            $user->save();
            return response()->json(['success' => true]);
        }
        return response()->json(['success' => false]);
    }

    public function banMember($id, Request $request)
    {
        $user = User::find($id);
        if ($user) {
            $user->banned = 1;
            $user->ban_reason = $request->reason;
            $user->save();
            return response()->json(['success' => true]);
        }
        return response()->json(['success' => false]);
    }

    public function unbanMember($id, Request $request)
    {
        $user = User::find($id);
        if ($user) {
            $user->banned = 0;
            $user->ban_reason = null;
            $user->save();
            return response()->json(['success' => true]);
        }
        return response()->json(['success' => false]);
    }

    // User search API for admin
    public function search(Request $request)
    {
        $query = $request->get('q');
        $page = $request->get('page', 1);
        $perPage = 20;
        
        if (!$query || strlen($query) < 2) {
            return response()->json([
                'data' => [],
                'has_more' => false
            ]);
        }
        
        $users = User::where('user_type', '!=', 1) // Exclude admins
            ->where(function($q) use ($query) {
                $q->where('account_holder', 'like', "%{$query}%")
                  ->orWhere('email', 'like', "%{$query}%")
                  ->orWhere('phone', 'like', "%{$query}%");
            })
            ->select('id', 'account_holder', 'email', 'phone')
            ->skip(($page - 1) * $perPage)
            ->take($perPage + 1)
            ->get();
            
        $hasMore = $users->count() > $perPage;
        if ($hasMore) {
            $users = $users->take($perPage);
        }
        
        return response()->json([
            'data' => $users->map(function($user) {
                return [
                    'id' => $user->id,
                    'account_holder' => $user->account_holder,
                    'email' => $user->email,
                    'phone' => $user->phone
                ];
            }),
            'has_more' => $hasMore
        ]);
    }

}
