<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Helpers\UserHelper;
use App\Models\Task;
use App\Models\TaskProof;
use App\Models\User;
use App\Models\SocialPlatform;
use App\Models\Country;
use App\Models\City;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class UserProfileController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display user profile page
     */
    public function index()
    {
        $user = auth()->user();
        $currency = $settings->currency ?? '₺';
        // Get user's created tasks (not completed tasks)
        $userTasks = Task::where('user_id', $user->id)
            ->with('socialPlatform')
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Get user's watch tasks
        $userWatchTasks = \App\Models\WatchTask::where('created_by', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Get user's website visit tasks
        $userWebsiteTasks = \App\Models\WebsiteVisitTask::where('created_by', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Get user's task proofs
        $userProofs = TaskProof::where('UserId', $user->id)
            ->with(['task', 'files'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Calculate statistics
        $totalTasks = $userTasks->count() + $userWatchTasks->count() + $userWebsiteTasks->count();
        $activeTasks = $userTasks->where('status', 1)->count() + 
                      $userWatchTasks->where('is_active', true)->count() + 
                      $userWebsiteTasks->where('is_active', true)->count();
        $approvedProofs = $userProofs->where('Status', 'approved')->count();
        
        // Calculate total earnings from approved proofs
        $totalEarnings = $userProofs->where('Status', 'approved')
            ->sum(function($proof) {
                return $proof->task ? $proof->task->reward : 0;
            });
        
        return view('user.profile', compact(
            'user',
            'userTasks', 
            'userWatchTasks',
            'userWebsiteTasks',
            'userProofs',
            'totalTasks',
            'activeTasks',
            'approvedProofs',
            'totalEarnings',
            'currency'
        ));
    }

    /**
     * Show edit profile form
     */
    public function edit()
    {
        $countries = Country::all();
        $cities = City::all();
        
        return view('user.edit-profile', compact('countries', 'cities'));
    }

    /**
     * Update user profile
     */
    public function update(Request $request)
{
    $user = auth()->user();
    
    try {
        // 1. Özelleştirilmiş validation mesajları
        $messages = [
            'username.required'               => __('Username is required.'),
            'username.unique'                 => __('This username is already taken.'),
            'email.required'                  => __('Email is required.'),
            'email.email'                     => __('Please enter a valid email address.'),
            'email.unique'                    => __('This email is already taken.'),
            'phone.max'                       => __('Phone number cannot exceed 20 characters.'),
            'country_id.exists'               => __('Selected country is invalid.'),
            'city_id.exists'                  => __('Selected city is invalid.'),
            'user_avatar.image'               => __('Avatar must be an image file.'),
            'user_avatar.mimes'               => __('Avatar must be a file of type: jpeg, png, jpg, gif.'),
            'user_avatar.max'                 => __('Avatar size cannot exceed 2MB.'),
            'account_holder_name.string'      => __('Account holder name must be a valid string.'),
            'account_holder_name.max'         => __('Account holder name cannot exceed 255 characters.'),
            'iban.string'                     => __('IBAN must be a valid string.'),
            'iban.max'                        => __('IBAN cannot exceed 34 characters.'),
            'bank_name.string'                => __('Bank name must be a valid string.'),
            'bank_name.max'                   => __('Bank name cannot exceed 255 characters.'),
        ];

        // 2. Gelen verilerin doğrulanması
        $validated = $request->validate([
            'username'             => 'required|string|max:255|unique:users,username,' . $user->id,
            'email'                => 'required|email|unique:users,email,' . $user->id,
            'phone'                => 'nullable|string|max:20',
            'country_id'           => 'nullable|exists:countries,id',
            'city_id'              => 'nullable|exists:cities,id',
            'user_avatar'          => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'account_holder_name'  => 'nullable|string|max:255',
            'iban'                 => 'nullable|string|max:34',
            'bank_name'            => 'nullable|string|max:255',
        ], $messages);
        
        // 3. Güncellenecek alanların seçilmesi
        $data = $request->only([
            'username',
            'email',
            'phone',
            'country_id',
            'city_id',
            'account_holder_name',
            'iban',
            'bank_name',
        ]);
        
        // 4. Avatar yükleme işlemi
        if ($request->hasFile('user_avatar')) {
            $avatar = $request->file('user_avatar');
            $filename = time() . '.' . $avatar->getClientOriginalExtension();
            $avatar->move(public_path('assets/images/avatars'), $filename);
            $data['user_avatar'] = $filename;
        }
        
        // 5. account_holder_name → account_holder alan eşleştirmesi
        if (!empty($data['account_holder_name'])) {
            $data['account_holder'] = $data['account_holder_name'];
            unset($data['account_holder_name']);
        }
        
        // 6. Kullanıcıyı güncelle
        $user->update($data);
        
        // 7. AJAX yanıtı
        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => __('Profile updated successfully'),
                'user' => [
                    'username'   => $user->username,
                    'email'      => $user->email,
                    'phone'      => $user->phone,
                    'avatar_url' => $user->user_avatar
                        ? asset('assets/images/avatars/' . $user->user_avatar)
                        : asset('assets/images/avatars/avatar-1.png'),
                ],
                'debug' => [
                    'request_data'     => $request->only(['iban', 'bank_name', 'account_holder_name']),
                    'validated_data'   => array_intersect_key(
                        $data,
                        array_flip(['iban', 'bank_name', 'account_holder'])
                    ),
                    'user_after_update'=> [
                        'iban'           => $user->fresh()->iban,
                        'bank_name'      => $user->fresh()->bank_name,
                        'account_holder' => $user->fresh()->account_holder,
                    ],
                ],
            ]);
        }
        
        // 8. Normal yönlendirme
        return redirect()->route('user.profile')
            ->with('success', __('Profile updated successfully'));
            
    } catch (\Illuminate\Validation\ValidationException $e) {
        if ($request->ajax()) {
            return response()->json([
                'success' => false,
                'message' => __('Validation failed'),
                'errors'  => $e->errors(),
            ], 422);
        }
        throw $e;
        
    } catch (\Exception $e) {
        if ($request->ajax()) {
            return response()->json([
                'success' => false,
                'message' => __('An error occurred: ') . $e->getMessage(),
            ], 500);
        }
        throw $e;
    }
}

    /**
     * Update user password
     */
    public function updatePassword(Request $request)
    {
        $user = auth()->user();
        
        try {
            $messages = [
                'current_password.required' => __('Current password is required.'),
                'new_password.required' => __('New password is required.'),
                'new_password.min' => __('New password must be at least 8 characters.'),
                'new_password.confirmed' => __('Password confirmation does not match.'),
                'new_password.regex' => __('Password must contain at least one uppercase letter, one lowercase letter, and one number.'),
            ];

            $validated = $request->validate([
                'current_password' => 'required',
                'new_password' => [
                    'required',
                    'min:8',
                    'confirmed',
                    'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).+$/'
                ],
            ], $messages);
            
            // Check if current password is correct
            if (!Hash::check($validated['current_password'], $user->password)) {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => __('Current password is incorrect'),
                        'errors' => ['current_password' => [__('Current password is incorrect')]]
                    ], 422);
                }
                return back()->withErrors(['current_password' => __('Current password is incorrect')]);
            }
            
            // Update password
            $user->password = Hash::make($validated['new_password']);
            $user->save();
            
            // AJAX response
            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => __('Password changed successfully')
                ]);
            }
            
            return redirect()->route('user.profile')
                ->with('success', __('Password changed successfully'));
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => __('Validation failed'),
                    'errors' => $e->errors()
                ], 422);
            }
            throw $e;
        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => __('An error occurred: ') . $e->getMessage()
                ], 500);
            }
            throw $e;
        }
    }
}