<?php

namespace App\Http\Controllers;

use App\Models\WatchTask;
use App\Models\WatchCompletion;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class WatchController extends Controller
{
    // İzleme görevleri listesi
    public function index()
{
    $user = Auth::user();
 $currency = $settings->currency ?? '₺';
    $watchTasks = WatchTask::with(['completions' => function($query) use ($user) {
        $query->where('user_id', $user->id);
    }])
    ->where('is_active', true)
    ->orderBy('created_at', 'desc')
    ->get();

    return view('watch.index', compact('watchTasks', 'user','currency'));
}

    // Video izleme sayfası
    public function watch($taskId)
    {
        $user = Auth::user();
        $watchTask = WatchTask::findOrFail($taskId);

        // Görev tamamlanabilir mi kontrol et
        if (!$watchTask->canBeCompleted()) {
            return redirect()->route('tasks.watch')->with('error', __('This task is no longer available.'));
        }

        // Kullanıcı daha önce tamamlamış mı?
        if ($watchTask->isCompletedByUser($user->id)) {
            return redirect()->route('tasks.watch')->with('info', __('You have already completed this task.'));
        }

        // İzleme kaydı oluştur veya güncelle
   $completion = WatchCompletion::updateOrCreate(
    [
        'watch_task_id' => $watchTask->id,
        'user_id'       => $user->id
    ],
    [
        'started_at'       => now(),
        'ip_address'       => request()->ip(),
        'user_agent'       => request()->userAgent(),
        'watched_duration' => 0,     // zaten eklemiştik
        'earned_amount'    => 0,     // <<< burayı ekledik
        'is_completed'     => false, // isterseniz bunu da koyabilirsiniz
    ]
);
return view('watch.player', [
    'watchTask' => $watchTask,
    'task'       => $watchTask,
    'completion' => $completion,
    'user'       => $user,
]);
    }

    // İzleme ilerlemesini güncelle (AJAX)
    public function updateProgress(Request $request)
    {
        $request->validate([
            'task_id' => 'required|exists:watch_tasks,id',
            'watched_duration' => 'required|integer|min:0'
        ]);

        $user = Auth::user();
        $watchTask = WatchTask::findOrFail($request->task_id);
        
        $completion = WatchCompletion::where('watch_task_id', $watchTask->id)
            ->where('user_id', $user->id)
            ->first();

        if (!$completion) {
            return response()->json(['success' => false, 'message' => 'Completion record not found.']);
        }

        // İzleme süresini güncelle
        $completion->watched_duration = max($completion->watched_duration, $request->watched_duration);
        $completion->save();

        // Görev tamamlandı mı kontrol et
        $isCompleted = $completion->watched_duration >= $watchTask->watch_duration;
        
        $response = [
            'success' => true,
            'watched_duration' => $completion->watched_duration,
            'required_duration' => $watchTask->watch_duration,
            'percentage' => min(100, ($completion->watched_duration / $watchTask->watch_duration) * 100),
            'is_completed' => $isCompleted
        ];

        if ($isCompleted && !$completion->is_completed) {
            $response['can_claim_reward'] = true;
        }

        return response()->json($response);
    }

    // Ödülü talep et
    public function claimReward(Request $request)
    {
        $request->validate([
            'task_id' => 'required|exists:watch_tasks,id'
        ]);

        $user = Auth::user();
        $watchTask = WatchTask::findOrFail($request->task_id);
        
        $completion = WatchCompletion::where('watch_task_id', $watchTask->id)
            ->where('user_id', $user->id)
            ->where('is_completed', false)
            ->first();

        if (!$completion) {
            return response()->json(['success' => false, 'message' => 'Completion record not found.']);
        }

        // İzleme süresi yeterli mi?
        if ($completion->watched_duration < $watchTask->watch_duration) {
            return response()->json(['success' => false, 'message' => 'Watch duration not sufficient.']);
        }

        DB::beginTransaction();
        try {
            // Ödülü kullanıcıya ekle
            $user->balance += $watchTask->reward_amount;
            $user->save();

            // Completion kaydını güncelle
            $completion->is_completed = true;
            $completion->completed_at = now();
            $completion->earned_amount = $watchTask->reward_amount;
            $completion->save();

            // Watch task completion sayısını artır
            $watchTask->increment('current_completions');

            // Transaction log kaydı
            \App\Models\TransactionLog::create([
                'user_id' => $user->id,
                'amount' => $watchTask->reward_amount,
                'type' => 'watch_reward',
                'description' => "Video izleme ödülü: {$watchTask->title}",
                'operation_type' => 'credit',
                'reference_type' => 'watch_task',
                'reference_id' => $watchTask->id,
                'balance_before' => $user->balance - $watchTask->reward_amount,
                'balance_after' => $user->balance
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => __('Congratulations! You earned :amount for watching the video.', ['amount' => $watchTask->reward_amount . '₺']),
                'earned_amount' => $watchTask->reward_amount,
                'new_balance' => $user->balance
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['success' => false, 'message' => 'An error occurred while processing reward.']);
        }
    }

    // Admin: Yeni izleme görevi oluştur
    public function create()
    {
        return view('watch.create');
    }

    // İzleme görevi kaydet (AJAX)
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'youtube_url' => 'required|url',
            'watch_duration' => 'required|integer|min:10|max:3600', // 10 saniye - 1 saat
            'reward_amount' => 'required|numeric|min:0.01|max:1000',
            'max_completions' => 'nullable|integer|min:0'
        ]);

        $user = Auth::user();
        $videoId = WatchTask::extractVideoId($request->youtube_url);
        
        if (!$videoId) {
            return response()->json([
                'success' => false,
                'message' => __('Invalid YouTube URL.')
            ]);
        }

        // Toplam maliyet hesapla
        $maxCompletions = $request->max_completions ?? 100; // Varsayılan 100
        $totalCost = $request->reward_amount * $maxCompletions;

        // Bakiye kontrolü
        if ($user->balance < $totalCost) {
            return response()->json([
                'success' => false,
                'message' => __('Insufficient balance. Required: :amount₺, Available: :balance₺', [
                    'amount' => $totalCost,
                    'balance' => $user->balance
                ])
            ]);
        }

        DB::beginTransaction();
        try {
            // Bakiyeden düş
            $balanceBefore = $user->balance;
            $user->balance -= $totalCost;
            $user->save();

            // Watch task oluştur
            $watchTask = WatchTask::create([
                'title' => $request->title,
                'description' => $request->description,
                'youtube_url' => $request->youtube_url,
                'youtube_video_id' => $videoId,
                'watch_duration' => $request->watch_duration,
                'reward_amount' => $request->reward_amount,
                'max_completions' => $maxCompletions,
                'created_by' => $user->id
            ]);

            // Transaction log kaydı
            \App\Models\TransactionLog::create([
                'user_id' => $user->id,
                'amount' => $totalCost,
                'type' => 'watch_task_creation',
                'description' => "Video izleme görevi oluşturuldu: {$request->title} ({$maxCompletions} tamamlanma x {$request->reward_amount}₺)",
                'operation_type' => 'debit',
                'reference_type' => 'watch_task',
                'reference_id' => $watchTask->id,
                'balance_before' => $balanceBefore,
                'balance_after' => $user->balance
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => __('Watch task created successfully. Total cost: :cost₺', ['cost' => $totalCost]),
                'task' => $watchTask,
                'new_balance' => $user->balance
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while creating the task.')
            ]);
        }
    }

    // Admin: Watch task analiz sayfası
    public function adminAnalytics()
    {
        $stats = [
            'total_tasks' => WatchTask::count(),
            'active_tasks' => WatchTask::where('is_active', true)->count(),
            'inactive_tasks' => WatchTask::where('is_active', false)->count(),
            'total_completions' => WatchCompletion::where('is_completed', true)->count(),
            'total_rewards_paid' => WatchCompletion::where('is_completed', true)->sum('earned_amount'),
            'pending_completions' => WatchCompletion::where('is_completed', false)->count(),
            'total_users' => WatchCompletion::distinct('user_id')->count(),
            'avg_completion_rate' => $this->getAverageCompletionRate(),
            'today_completions' => WatchCompletion::where('is_completed', true)->whereDate('completed_at', today())->count(),
            'this_week_completions' => WatchCompletion::where('is_completed', true)->whereBetween('completed_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
            'top_creators' => $this->getTopCreators(),
            'recent_activities' => $this->getRecentActivities()
        ];

        $settings = app('settings');
        return view('admin.watch.analytics', compact('stats', 'settings'));
    }

    // User: Kendi watch task analiz sayfası
    public function userAnalytics()
    {
        $user = Auth::user();
        
        $stats = [
            'my_total_tasks' => WatchTask::where('created_by', $user->id)->count(),
            'my_active_tasks' => WatchTask::where('created_by', $user->id)->where('is_active', true)->count(),
            'my_inactive_tasks' => WatchTask::where('created_by', $user->id)->where('is_active', false)->count(),
            'my_total_completions' => WatchCompletion::whereHas('watchTask', function($q) use ($user) {
                $q->where('created_by', $user->id);
            })->where('is_completed', true)->count(),
            'my_total_rewards_paid' => WatchCompletion::whereHas('watchTask', function($q) use ($user) {
                $q->where('created_by', $user->id);
            })->where('is_completed', true)->sum('earned_amount'),
            'my_pending_completions' => WatchCompletion::whereHas('watchTask', function($q) use ($user) {
                $q->where('created_by', $user->id);
            })->where('is_completed', false)->count(),
            'my_total_investment' => WatchTask::where('created_by', $user->id)->sum(DB::raw('reward_amount * max_completions')),
            'my_roi' => $this->getUserROI($user->id),
            'my_avg_completion_rate' => $this->getUserAverageCompletionRate($user->id),
            'my_today_completions' => WatchCompletion::whereHas('watchTask', function($q) use ($user) {
                $q->where('created_by', $user->id);
            })->where('is_completed', true)->whereDate('completed_at', today())->count(),
            'my_this_week_completions' => WatchCompletion::whereHas('watchTask', function($q) use ($user) {
                $q->where('created_by', $user->id);
            })->where('is_completed', true)->whereBetween('completed_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
            'my_recent_activities' => $this->getUserRecentActivities($user->id)
        ];

        $settings = app('settings');
        return view('user.watch.analytics', compact('stats', 'settings', 'user'));
    }

    // Admin: Watch tasks DataTable
    public function adminDataTable(Request $request)
    {
        $query = WatchTask::with(['creator', 'completions']);

        // Search functionality
        if ($request->has('search') && $request->search['value']) {
            $searchValue = $request->search['value'];
            $query->where('title', 'like', "%{$searchValue}%")
                  ->orWhere('description', 'like', "%{$searchValue}%")
                  ->orWhereHas('creator', function($q) use ($searchValue) {
                      $q->where('account_holder', 'like', "%{$searchValue}%");
                  });
        }

        // Total records
        $totalRecords = $query->count();

        // Pagination
        $start = $request->start ?? 0;
        $length = $request->length ?? 10;
        
        $tasks = $query->skip($start)->take($length)
            ->orderBy('created_at', 'desc')
            ->get();

        $data = [];
        foreach ($tasks as $task) {
            $data[] = [
                'id' => $task->id,
                'title' => $task->title,
                'creator' => $task->creator->account_holder,
                'reward_amount' => $task->reward_amount,
                'watch_duration' => $task->formatted_duration,
                'max_completions' => $task->max_completions,
                'current_completions' => $task->current_completions,
                'total_cost' => $task->reward_amount * $task->max_completions,
                'is_active' => $task->is_active,
                'created_at' => $task->created_at->format('d.m.Y H:i'),
                'actions' => $this->getAdminTaskActionButtons($task)
            ];
        }

        return response()->json([
            'draw' => $request->draw,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $totalRecords,
            'data' => $data
        ]);
    }

    private function getAdminTaskActionButtons($task)
    {
        $buttons = '<button class="btn btn-sm btn-info view-task" data-id="'.$task->id.'">
                        <i class="bx bx-show"></i> '.__('View').'
                    </button>';
        
        if ($task->is_active) {
            $buttons .= ' <button class="btn btn-sm btn-warning deactivate-task" data-id="'.$task->id.'">
                            <i class="bx bx-pause"></i> '.__('Deactivate').'
                        </button>';
        } else {
            $buttons .= ' <button class="btn btn-sm btn-success activate-task" data-id="'.$task->id.'">
                            <i class="bx bx-play"></i> '.__('Activate').'
                        </button>';
        }
        
        return $buttons;
    }

    // User: Kendi watch tasks DataTable
    public function userDataTable(Request $request)
    {
        $user = Auth::user();
        $query = WatchTask::with(['completions'])->where('created_by', $user->id);

        // Search functionality
        if ($request->has('search') && $request->search['value']) {
            $searchValue = $request->search['value'];
            $query->where('title', 'like', "%{$searchValue}%")
                  ->orWhere('description', 'like', "%{$searchValue}%");
        }

        // Total records
        $totalRecords = $query->count();

        // Pagination
        $start = $request->start ?? 0;
        $length = $request->length ?? 10;
        
        $tasks = $query->skip($start)->take($length)
            ->orderBy('created_at', 'desc')
            ->get();

        $data = [];
        foreach ($tasks as $task) {
            $completionRate = $task->max_completions > 0 ? ($task->current_completions / $task->max_completions) * 100 : 0;
            $totalCost = $task->reward_amount * $task->max_completions;
            $totalPaid = $task->current_completions * $task->reward_amount;
            $remaining = $totalCost - $totalPaid;

            $data[] = [
                'id' => $task->id,
                'title' => $task->title,
                'reward_amount' => $task->reward_amount,
                'watch_duration' => $task->formatted_duration,
                'max_completions' => $task->max_completions,
                'current_completions' => $task->current_completions,
                'completion_rate' => round($completionRate, 1),
                'total_cost' => $totalCost,
                'total_paid' => $totalPaid,
                'remaining_budget' => $remaining,
                'is_active' => $task->is_active,
                'created_at' => $task->created_at->format('d.m.Y H:i'),
                'actions' => $this->getUserTaskActionButtons($task)
            ];
        }

        return response()->json([
            'draw' => $request->draw,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $totalRecords,
            'data' => $data
        ]);
    }

    private function getUserTaskActionButtons($task)
    {
        $buttons = '<button class="btn btn-sm btn-info view-task" data-id="'.$task->id.'">
                        <i class="bx bx-show"></i> '.__('View').'
                    </button>';
        
        $buttons .= ' <button class="btn btn-sm btn-primary edit-task" data-id="'.$task->id.'">
                        <i class="bx bx-edit"></i> '.__('Edit').'
                    </button>';
        
        if ($task->is_active) {
            $buttons .= ' <button class="btn btn-sm btn-warning deactivate-task" data-id="'.$task->id.'">
                            <i class="bx bx-pause"></i> '.__('Deactivate').'
                        </button>';
        } else {
            $buttons .= ' <button class="btn btn-sm btn-success activate-task" data-id="'.$task->id.'">
                            <i class="bx bx-play"></i> '.__('Activate').'
                        </button>';
        }
        
        if ($task->current_completions == 0) {
            $buttons .= ' <button class="btn btn-sm btn-danger delete-task" data-id="'.$task->id.'">
                            <i class="bx bx-trash"></i> '.__('Delete').'
                        </button>';
        }
        
        return $buttons;
    }

    // Helper methods for analytics
    private function getAverageCompletionRate()
    {
        $tasks = WatchTask::select('id', 'max_completions', DB::raw('(SELECT COUNT(*) FROM watch_completions WHERE watch_task_id = watch_tasks.id AND is_completed = 1) as completions_count'))
            ->get();
            
        if ($tasks->isEmpty()) return 0;
        
        $totalRate = 0;
        foreach ($tasks as $task) {
            if ($task->max_completions > 0) {
                $totalRate += ($task->completions_count / $task->max_completions) * 100;
            }
        }
        
        return round($totalRate / $tasks->count(), 1);
    }

    private function getUserAverageCompletionRate($userId)
    {
        $tasks = WatchTask::where('created_by', $userId)
            ->select('id', 'max_completions', DB::raw('(SELECT COUNT(*) FROM watch_completions WHERE watch_task_id = watch_tasks.id AND is_completed = 1) as completions_count'))
            ->get();
            
        if ($tasks->isEmpty()) return 0;
        
        $totalRate = 0;
        foreach ($tasks as $task) {
            if ($task->max_completions > 0) {
                $totalRate += ($task->completions_count / $task->max_completions) * 100;
            }
        }
        
        return round($totalRate / $tasks->count(), 1);
    }

    private function getUserROI($userId)
    {
        $totalInvestment = WatchTask::where('created_by', $userId)->sum(DB::raw('reward_amount * max_completions'));
        $totalReturns = WatchCompletion::whereHas('watchTask', function($q) use ($userId) {
            $q->where('created_by', $userId);
        })->where('is_completed', true)->sum('earned_amount');
        
        if ($totalInvestment == 0) return 0;
        
        return round((($totalReturns - $totalInvestment) / $totalInvestment) * 100, 1);
    }

    private function getTopCreators()
    {
        return WatchTask::select('created_by', DB::raw('COUNT(watch_completions.id) as completions_count'))
            ->leftJoin('watch_completions', function($join) {
                $join->on('watch_tasks.id', '=', 'watch_completions.watch_task_id')
                     ->where('watch_completions.is_completed', '=', 1);
            })
            ->with('creator:id,account_holder')
            ->groupBy('created_by')
            ->orderBy('completions_count', 'desc')
            ->limit(5)
            ->get();
    }

    private function getRecentActivities()
    {
        return WatchCompletion::with(['watchTask:id,title', 'user:id,account_holder'])
            ->where('is_completed', true)
            ->orderBy('completed_at', 'desc')
            ->limit(10)
            ->get();
    }

    private function getUserRecentActivities($userId)
    {
        return WatchCompletion::with(['watchTask:id,title', 'user:id,account_holder'])
            ->whereHas('watchTask', function($q) use ($userId) {
                $q->where('created_by', $userId);
            })
            ->where('is_completed', true)
            ->orderBy('completed_at', 'desc')
            ->limit(10)
            ->get();
    }

    // Kullanıcının kendi görevleri sayfası
    public function myTasks()
    {
        $user = Auth::user();
        return view('user.watch.my-tasks', compact('user'));
    }

    // Görev düzenleme için veri getir
    public function editTask(WatchTask $task)
    {
        // Yetki kontrolü
        if ($task->created_by != auth()->id() && !auth()->user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Bu işlem için yetkiniz yok.'], 403);
        }

        return response()->json([
            'success' => true,
            'task' => [
                'id'                  => $task->id,
                'title'               => $task->title,
                'description'         => $task->description,
                'youtube_url'         => $task->youtube_url,
                'youtube_video_id'    => $task->youtube_video_id,
                'watch_duration'      => $task->watch_duration,
                'formatted_duration'  => $task->formatted_duration,
                'reward_amount'       => $task->reward_amount,
                'max_completions'     => $task->max_completions,
                'current_completions' => $task->current_completions,
                'is_active'           => $task->is_active,
                'created_at'          => $task->created_at->format('d.m.Y H:i'),
            ]
        ]);
    }

    // Görev güncelle
    public function updateTask(Request $request, WatchTask $task)
    {
        // Yetki kontrolü
        if ($task->created_by != auth()->id() && !auth()->user()->isAdmin()) {
            return response()->json(['success' => false, 'message' => 'Bu işlem için yetkiniz yok.'], 403);
        }

        $data = $request->validate([
            'title'         => 'required|string|max:255',
            'description'   => 'nullable|string',
            'watch_duration'=> 'required|integer|min:10|max:3600',
            'reward_amount' => 'required|numeric|min:0.01',
        ]);

        $task->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Görev başarıyla güncellendi.',
            'task'    => $task
        ]);
    }

    public function destroy(WatchTask $task)
{
    // Yetki kontrolü
    if ($task->created_by != auth()->id() && ! auth()->user()->isAdmin()) {
        return response()->json([
            'success' => false,
            'message' => 'Bu işlem için yetkiniz yok.'
        ], 403);
    }

    // Eğer bu göreve ait en az bir completion varsa -> silme, pasif yap
    if ($task->completions()->exists()) {
        $task->is_active = false;
        $task->save();

        return response()->json([
            'success' => true,
            'message' => 'Bu görevde tamamlanma kaydı bulunduğu için silinemedi, ancak pasif duruma alındı.'
        ]);
    }

    // İzleme kaydı yoksa, refund ve silme işlemi
    $refundAmount = 0;

    DB::transaction(function() use ($task, &$refundAmount) {
        // Tamamlanmamışsa iade
        if ($task->current_completions < $task->max_completions) {
            $remaining    = $task->max_completions - $task->current_completions;
            $refundAmount = $remaining * $task->reward_amount;

            $creator = $task->creator;
            $creator->balance += $refundAmount;
            $creator->save();

            \App\Models\BalanceLog::create([
                'userId'         => $creator->id,
                'balance_before' => $creator->balance - $refundAmount,
                'balance'        => $refundAmount,
                'balance_after'  => $creator->balance,
                'description'    => "Watch task silme iadesi: {$task->title}",
                'type'           => 'watch_task_refund',
                'reference_type' => 'watch_task',
                'reference_id'   => $task->id,
                'created_at'     => now(),
                'updated_at'     => now(),
            ]);
        }

        // Kayıt silinir
        $task->delete();
    });

    return response()->json([
        'success'       => true,
        'message'       => 'Görev silindi' . ($refundAmount > 0 ? " ve {$refundAmount}₺ iade edildi" : ''),
        'refund_amount' => $refundAmount,
    ]);
}
}