<?php

namespace App\Http\Controllers;

use App\Models\WebsiteVisitTask;
use App\Models\WebsiteVisit;
use App\Models\WebsiteVisitProof;
use App\Models\WebsiteVisitProofFile;
use App\Models\Country;
use App\Models\City;
use App\Models\BalanceLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class WebsiteVisitController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display available website visit tasks
     */
    public function index()
    {
        $user = Auth::user();
        
        $tasksQuery = WebsiteVisitTask::active()
            ->where(function($query) {
                $query->whereNull('expires_at')
                      ->orWhere('expires_at', '>', now());
            })
            ->with(['country', 'city', 'creator'])
            ->orderBy('created_at', 'desc');

        // Admin tüm görevleri görebilir, normal kullanıcı sadece kendine uygun olanları
        \Log::info('User admin check:', [
            'user_id' => $user->id,
            'user_type' => $user->user_type,
            'is_admin' => $user->isAdmin(),
            'TYPE_ADMIN' => User::TYPE_ADMIN
        ]);
        
        if (!$user->isAdmin()) {
            $tasksQuery->forUser($user);
        }

        $tasks = $tasksQuery->get();

        // Add user-specific data to each task
        $tasks->each(function($task) use ($user) {
            $task->user_can_complete = $task->canUserComplete($user);
            $task->user_today_count = $task->visits()
                ->where('user_id', $user->id)
                ->whereDate('created_at', today())
                ->where('status', 'completed')
                ->count();
            $task->user_total_count = $task->visits()
                ->where('user_id', $user->id)
                ->where('status', 'completed')
                ->count();
        });

        return view('website-visits.index', compact('tasks', 'user'));
    }

    /**
     * Show task creation form (admin only)
     */
    public function create()
    {
        $this->authorize('create', WebsiteVisitTask::class);
        
        $countries = Country::all();
        $cities = City::all();
        
        return view('website-visits.create', compact('countries', 'cities'));
    }

    /**
     * Store new website visit task
     */
    public function store(Request $request)
    {
        $this->authorize('create', WebsiteVisitTask::class);
        
        $messages = [
            'title.required' => __('Task title is required.'),
            'website_url.required' => __('Website URL is required.'),
            'website_url.url' => __('Please enter a valid URL.'),
            'required_time.required' => __('Required time is required.'),
            'required_time.min' => __('Required time must be at least 10 seconds.'),
            'reward_amount.required' => __('Reward amount is required.'),
            'reward_amount.min' => __('Reward amount must be greater than 0.'),
            'max_completions.required' => __('Maximum completions is required.'),
            'user_daily_limit.required' => __('User daily limit is required.'),
            'user_total_limit.required' => __('User total limit is required.'),
        ];

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'website_url' => 'required|url',
            'visit_method' => 'required|in:direct,google_search',
            'search_keywords' => 'required_if:visit_method,google_search|string|max:255',
            'required_time' => 'required|integer|min:10|max:3600',
            'reward_amount' => 'required|numeric|min:0.01',
            'max_completions' => 'required|integer|min:1',
            'user_daily_limit' => 'required|integer|min:1|max:50',
            'user_total_limit' => 'required|integer|min:1',
            'target_country_id' => 'nullable|integer',
            'target_city_id' => 'nullable|integer',
            'target_gender' => 'required|in:0,1,2',
            'expires_at' => 'nullable|date|after:now',
        ], $messages);

        // Add missing fields manually from request
        $validated['created_by'] = Auth::id();
        $validated['target_country_id'] = $request->input('target_country_id', 0);
        $validated['target_city_id'] = $request->input('target_city_id', 0);
        
        // Convert empty strings to 0
        if (empty($validated['target_country_id']) || $validated['target_country_id'] === '') {
            $validated['target_country_id'] = 0;
        }
        if (empty($validated['target_city_id']) || $validated['target_city_id'] === '') {
            $validated['target_city_id'] = 0;
        }

        $task = WebsiteVisitTask::create($validated);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => __('Website visit task created successfully.'),
                'task' => $task
            ]);
        }

        return redirect()->route('website-visits.index')
            ->with('success', __('Website visit task created successfully.'));
    }

    /**
     * Start a website visit task
     */
    public function startVisit(Request $request, WebsiteVisitTask $task)
    {
        $user = Auth::user();

        // Check if user can complete this task
        if (!$task->canUserComplete($user)) {
            return response()->json([
                'success' => false,
                'message' => __('You have reached the limit for this task.')
            ], 403);
        }

        // Check if user has an active visit for this task (sadece tamamlanmamış ziyaretler)
        $activeVisit = WebsiteVisit::where('task_id', $task->id)
            ->where('user_id', $user->id)
            ->whereIn('status', ['started', 'in_progress'])
            ->where('created_at', '>', now()->subHours(2)) // 2 saatten eski ziyaretleri aktif sayma
            ->first();

        if ($activeVisit) {
            // Return active visit info for continuation
            $visitUrl = $this->generateVisitUrl($task, $activeVisit);
            
            return response()->json([
                'success' => true,
                'has_active_visit' => true,
                'message' => __('You have an active visit for this task.'),
                'visit_id' => $activeVisit->id,
                'visit_url' => $visitUrl,
                'required_time' => $task->required_time,
                'verification_token' => $activeVisit->verification_token,
                'start_time' => $activeVisit->start_time->timestamp
            ]);
        }

        // Create new visit record with proper timezone
        $startTime = now();
        
        $visit = WebsiteVisit::create([
            'task_id' => $task->id,
            'user_id' => $user->id,
            'session_id' => session()->getId(),
            'start_time' => $startTime,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'referrer' => $request->header('referer'),
            'status' => 'started',
            'verification_token' => Str::random(32)
        ]);
        
        // Debug log
        \Log::info('Visit started:', [
            'visit_id' => $visit->id,
            'start_time' => $startTime->toISOString(),
            'start_time_timestamp' => $startTime->timestamp
        ]);

        // Generate visit URL based on method
        $visitUrl = $this->generateVisitUrl($task, $visit);

        return response()->json([
            'success' => true,
            'has_active_visit' => false,
            'message' => __('Visit started successfully.'),
            'visit_id' => $visit->id,
            'visit_url' => $visitUrl,
            'required_time' => $task->required_time,
            'verification_token' => $visit->verification_token,
            'start_time' => $startTime->timestamp
        ]);
    }

    /**
     * Track visit progress
     */
    public function trackVisit(Request $request, WebsiteVisit $visit)
    {
        $user = Auth::user();

        if ($visit->user_id !== $user->id) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        // Update visit status and track activity
        $visit->update([
            'status' => 'in_progress',
            'user_agent' => $request->userAgent(), // Track if user agent changes
        ]);

        // Calculate elapsed time correctly
        $elapsedTime = 0;
        if ($visit->start_time) {
            $startTime = \Carbon\Carbon::parse($visit->start_time);
            $currentTime = now();
            $elapsedTime = $currentTime->timestamp - $startTime->timestamp;
            
            // Debug log
            \Log::info('TRACK VISIT - Time calculation debug:', [
                'visit_id' => $visit->id,
                'start_time_raw' => $visit->start_time,
                'start_time_parsed' => $startTime->toISOString(),
                'current_time' => $currentTime->toISOString(),
                'timestamp_diff' => $elapsedTime,
                'start_timestamp' => $startTime->timestamp,
                'current_timestamp' => $currentTime->timestamp
            ]);
            
            // Ensure positive value
            $elapsedTime = max(0, $elapsedTime);
        }

        return response()->json([
            'success' => true,
            'elapsed_time' => $elapsedTime,
            'required_time' => $visit->task->required_time,
            'progress_percentage' => min(($elapsedTime / $visit->task->required_time) * 100, 100),
            'window_check' => true,
            'is_completed' => $elapsedTime >= $visit->task->required_time,
            'debug' => [
                'start_time' => $visit->start_time,
                'now' => now(),
                'calculated_elapsed' => $elapsedTime
            ]
        ]);
    }

    /**
     * Complete visit and give reward
     */
    public function completeVisit(Request $request, WebsiteVisit $visit)
    {
        $user = Auth::user();

        if ($visit->user_id !== $user->id) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        if ($visit->status === 'completed') {
            return response()->json(['success' => false, 'message' => 'Visit already completed'], 400);
        }

        // Frontend'den gelen elapsed_time'ı kullan, yoksa hesapla
        $elapsedTime = $request->input('elapsed_time', 0);
        
        // Eğer frontend'den gelmemişse sunucu tarafında hesapla
        if (!$elapsedTime && $visit->start_time) {
            $startTime = \Carbon\Carbon::parse($visit->start_time);
            $currentTime = now();
            $elapsedTime = $currentTime->timestamp - $startTime->timestamp;
            $elapsedTime = max(0, $elapsedTime);
        }
        
        // Debug log for time check
        \Log::info('COMPLETE VISIT - Time check:', [
            'visit_id' => $visit->id,
            'elapsed_time' => $elapsedTime,
            'required_time' => $visit->task->required_time,
            'is_sufficient' => $elapsedTime >= $visit->task->required_time,
            'start_time' => $visit->start_time,
            'current_time' => now(),
            'start_timestamp' => \Carbon\Carbon::parse($visit->start_time)->timestamp,
            'current_timestamp' => now()->timestamp
        ]);
        
        // Zaman kontrolünü tekrar aktif et
        if ($elapsedTime < $visit->task->required_time) {
            return response()->json([
                'success' => false,
                'message' => __('You need to spend at least :time seconds on the website.', [
                    'time' => $visit->task->required_time
                ]),
                'debug' => [
                    'elapsed_time' => $elapsedTime,
                    'required_time' => $visit->task->required_time,
                    'start_time' => $visit->start_time,
                    'current_time' => now()
                ]
            ], 400);
        }

        // Additional fraud checks
        $fraudChecks = $this->performFraudChecks($visit, $request);
        if (!$fraudChecks['passed']) {
            return response()->json([
                'success' => false,
                'message' => $fraudChecks['message']
            ], 400);
        }

        // Mark visit as completed with frontend duration
        \Log::info('About to call markAsCompleted:', [
            'visit_id' => $visit->id,
            'current_status' => $visit->status,
            'elapsed_time' => $elapsedTime
        ]);
        
        $visit->markAsCompleted($elapsedTime);
        
        // Reload to get updated duration
        $visit->refresh();
        
        \Log::info('After markAsCompleted and refresh:', [
            'visit_id' => $visit->id,
            'status_after_refresh' => $visit->status,
            'duration_after_refresh' => $visit->duration
        ]);

        // Give reward to user only if proof is not required
        if (!$visit->reward_given && !$this->requiresProof($visit->task)) {
            $user->balance += $visit->task->reward_amount;
            $user->save();

            // Log balance change
            BalanceLog::create([
                'userId' => $user->id,
                'TaskId' => $visit->id,
                'balance' => $user->balance,
                'balance_before' => $user->balance - $visit->task->reward_amount,
                'added_user' => $user->id
            ]);

            $visit->update(['reward_given' => true]);
        }

        $requiresProof = $this->requiresProof($visit->task);
        
        return response()->json([
            'success' => true,
            'message' => $requiresProof 
                ? __('Visit completed! Please submit proof to receive your reward.')
                : __('Visit completed successfully! You earned :amount credits.', ['amount' => $visit->task->reward_amount]),
            'reward' => $visit->task->reward_amount,
            'new_balance' => $user->balance,
            'requires_proof' => $requiresProof,
            'visit_id' => $visit->id
        ]);
    }

    /**
     * Perform fraud detection checks
     */
    private function performFraudChecks(WebsiteVisit $visit, Request $request)
    {
        // Check if too many visits from same IP in short time
        $recentVisits = WebsiteVisit::where('ip_address', $request->ip())
            ->where('created_at', '>', now()->subHour())
            ->where('status', 'completed')
            ->count();

        if ($recentVisits > 10) {
            return [
                'passed' => false,
                'message' => __('Too many visits from your IP address. Please try again later.')
            ];
        }

        // Frontend'den gelen elapsed_time'ı kullan
        $frontendElapsed = request()->input('elapsed_time', 0);
        $actualDuration = $frontendElapsed > 0 ? $frontendElapsed : ($visit->start_time ? now()->diffInSeconds($visit->start_time) : 0);
        $requiredTime = $visit->task->required_time;
        
        if ($actualDuration < ($requiredTime * 0.9)) { // Allow 10% tolerance
            return [
                'passed' => false,
                'message' => __('Visit duration is too short. Please spend the required time on the website.')
            ];
        }

        // Check if user agent changed during visit
        if ($visit->user_agent !== $request->userAgent()) {
            return [
                'passed' => false,
                'message' => __('Browser or device changed during visit. Please complete the task in the same browser.')
            ];
        }

        return ['passed' => true];
    }

    /**
     * Check if task requires proof submission
     */
    private function requiresProof(WebsiteVisitTask $task)
    {
        // Require proof for high-value tasks or suspicious patterns
        return $task->reward_amount > 1.0 || $task->required_time > 300; // 5 minutes
    }

    /**
     * Generate visit URL based on task method
     */
    private function generateVisitUrl(WebsiteVisitTask $task, WebsiteVisit $visit)
    {
        if ($task->visit_method === 'google_search') {
            $searchQuery = urlencode($task->search_keywords . ' site:' . parse_url($task->website_url, PHP_URL_HOST));
            return "https://www.google.com/search?q=" . $searchQuery;
        }

        return $task->website_url;
    }

    /**
     * Admin: List all website visit tasks
     */
    public function adminIndex()
    {
        $this->authorize('viewAny', WebsiteVisitTask::class);
        
        $tasks = WebsiteVisitTask::with(['country', 'city', 'creator'])
            ->withCount(['visits', 'completedVisits'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('website-visits.admin.index', compact('tasks'));
    }

    /**
     * Admin: Show task statistics
     */
    public function show(WebsiteVisitTask $task)
    {
        $this->authorize('view', $task);
        
        $task->load(['country', 'city', 'creator']);
        
        $stats = [
            'total_visits' => $task->visits()->count(),
            'completed_visits' => $task->completedVisits()->count(),
            'total_rewards_given' => $task->visits()->where('reward_given', true)->sum('task.reward_amount'),
            'unique_users' => $task->visits()->distinct('user_id')->count(),
            'avg_duration' => $task->completedVisits()->avg('duration'),
            'today_visits' => $task->visits()->whereDate('created_at', today())->count(),
        ];

        $recentVisits = $task->visits()
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('website-visits.show', compact('task', 'stats', 'recentVisits'));
    }

    /**
     * Submit proof for website visit
     */
    public function submitProof(Request $request)
    {
        $request->validate([
            'visit_id' => 'required|exists:website_visits,id',
            'description' => 'required|string|max:1000',
            'proofs' => 'required|array|min:1',
            'proofs.*' => 'required|file|mimes:jpeg,jpg,png,gif,webp|max:5120' // 5MB max
        ]);

        $visit = WebsiteVisit::findOrFail($request->visit_id);
        $user = Auth::user();

        if ($visit->user_id !== $user->id) {
            return response()->json(['success' => false, 'message' => __('Unauthorized')], 403);
        }

        // Süre kontrolü yap - proof submit edildiğinde görevi tamamla
        $elapsedTime = 0;
        if ($visit->start_time) {
            $startTime = \Carbon\Carbon::parse($visit->start_time);
            $currentTime = now();
            $elapsedTime = $currentTime->timestamp - $startTime->timestamp;
            $elapsedTime = max(0, $elapsedTime);
        }

        \Log::info('Submit proof - checking visit:', [
            'visit_id' => $visit->id,
            'current_status' => $visit->status,
            'elapsed_time' => $elapsedTime,
            'required_time' => $visit->task->required_time,
            'is_sufficient' => $elapsedTime >= $visit->task->required_time
        ]);

        // Eğer süre yeterli ve görev henüz tamamlanmamışsa, şimdi tamamla
        if ($elapsedTime >= $visit->task->required_time && $visit->status !== 'completed') {
            $visit->markAsCompleted($elapsedTime);
            $visit->refresh();
            
            \Log::info('Visit completed during proof submission:', [
                'visit_id' => $visit->id,
                'new_status' => $visit->status
            ]);
        }

        // Şimdi status kontrolü yap
        if ($visit->status !== 'completed') {
            return response()->json([
                'success' => false, 
                'message' => __('Visit must be completed first. Please spend at least :time seconds on the website.', [
                    'time' => $visit->task->required_time
                ])
            ], 400);
        }

        // Check if proof already exists
        $existingProof = WebsiteVisitProof::where('visit_id', $visit->id)->first();
        if ($existingProof) {
            return response()->json(['success' => false, 'message' => __('Proof already submitted for this visit')], 400);
        }

        // Create proof record
        $proof = WebsiteVisitProof::create([
            'visit_id' => $visit->id,
            'user_id' => $user->id,
            'description' => $request->description,
            'status' => 0 // pending
        ]);

        // Handle file uploads
        if ($request->hasFile('proofs')) {
            foreach ($request->file('proofs') as $file) {
                if ($file->isValid()) {
                    $extension = $file->getClientOriginalExtension();
                    $fileName = uniqid() . '.' . $extension;
                    
                    // Create directory if it doesn't exist
                    $uploadPath = public_path('assets/images/websiteVisitProofs');
                    if (!file_exists($uploadPath)) {
                        mkdir($uploadPath, 0755, true);
                    }
                    
                    // Get file info before moving
                    $fileSize = $file->getSize();
                    $mimeType = $file->getMimeType();
                    
                    // Move file
                    $file->move($uploadPath, $fileName);

                    WebsiteVisitProofFile::create([
                        'proof_id' => $proof->id,
                        'file_name' => $fileName,
                        'file_path' => 'assets/images/websiteVisitProofs/' . $fileName,
                        'file_type' => $mimeType,
                        'file_size' => $fileSize
                    ]);
                }
            }
        }

        return response()->json([
            'success' => true,
            'message' => __('Proof submitted successfully! It will be reviewed by administrators.')
        ]);
    }

    /**
     * Handle window close detection
     */
    public function handleWindowClose(Request $request)
    {
        $visitId = $request->input('visit_id');
        $visit = WebsiteVisit::find($visitId);
        
        if (!$visit || $visit->user_id !== Auth::id()) {
            return response()->json(['success' => false]);
        }

        // If visit is not completed and window was closed early, mark as failed
        if ($visit->status !== 'completed') {
            $elapsedTime = 0;
            if ($visit->start_time) {
                $startTime = \Carbon\Carbon::parse($visit->start_time);
                $currentTime = now();
                $elapsedTime = $currentTime->timestamp - $startTime->timestamp;
                $elapsedTime = max(0, $elapsedTime);
            }
            
            if ($elapsedTime < $visit->task->required_time) {
                $visit->update([
                    'status' => 'failed',
                    'end_time' => now(),
                    'duration' => $elapsedTime
                ]);
            }
        }

        return response()->json(['success' => true]);
    }

    /**
     * Cancel a website visit task
     */
    public function cancelVisit(Request $request, WebsiteVisit $visit)
    {
        $user = Auth::user();
        
        if ($visit->user_id !== $user->id) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        // Mark visit as failed
        $elapsedTime = 0;
        if ($visit->start_time) {
            $startTime = \Carbon\Carbon::parse($visit->start_time);
            $currentTime = now();
            $elapsedTime = $currentTime->timestamp - $startTime->timestamp;
            $elapsedTime = max(0, $elapsedTime);
        }
        
        $visit->update([
            'status' => 'failed',
            'end_time' => now(),
            'duration' => $elapsedTime
        ]);

        \Log::info('Visit cancelled by user:', [
            'visit_id' => $visit->id,
            'user_id' => $user->id,
            'elapsed_time' => $elapsedTime
        ]);

        return response()->json([
            'success' => true,
            'message' => __('Task cancelled successfully.')
        ]);
    }

    /**
     * Get website visit proofs for admin review
     */
    public function getProofs()
    {
        $this->authorize('viewAny', WebsiteVisitTask::class);
        
        $proofs = WebsiteVisitProof::with(['visit.task', 'user', 'files'])
            ->pending()
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('website-visits.proofs.index', compact('proofs'));
    }

    /**
     * Approve website visit proof
     */
    public function approveProof(Request $request, $proofId)
    {
        $this->authorize('viewAny', WebsiteVisitTask::class);
        
        $proof = WebsiteVisitProof::findOrFail($proofId);
        $visit = $proof->visit;
        $user = $visit->user;

        // Approve proof
        $proof->update([
            'status' => 1,
            'reviewed_by' => Auth::id(),
            'reviewed_at' => now()
        ]);

        // Give reward if not already given
        if (!$visit->reward_given) {
            $user->balance += $visit->task->reward_amount;
            $user->save();

            // Log balance change
            BalanceLog::create([
                'userId' => $user->id,
                'TaskId' => $visit->id,
                'balance' => $user->balance,
                'balance_before' => $user->balance - $visit->task->reward_amount,
                'added_user' => Auth::id()
            ]);

            $visit->update(['reward_given' => true]);
        }

        return response()->json([
            'success' => true,
            'message' => __('Proof approved and reward given to user.')
        ]);
    }

    /**
     * Deny website visit proof
     */
    public function denyProof(Request $request, $proofId)
    {
        $this->authorize('viewAny', WebsiteVisitTask::class);
        
        $request->validate([
            'reason' => 'required|string|max:500'
        ]);

        $proof = WebsiteVisitProof::findOrFail($proofId);
        
        $proof->update([
            'status' => 2,
            'reason' => $request->reason,
            'reviewed_by' => Auth::id(),
            'reviewed_at' => now()
        ]);

        return response()->json([
            'success' => true,
            'message' => __('Proof denied.')
        ]);
    }
}