<?php

namespace App\Http\Controllers;

use App\Models\WithdrawalRequest;
use App\Models\BalanceLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class WithdrawalController extends Controller
{
    // Kullanıcı çekim talepleri
    public function index()
    {
        $user = Auth::user();
        $withdrawals = WithdrawalRequest::where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Settings'den dinamik değerleri al
        $settings = \App\Models\Setting::first();
        $currency = $settings->currency ?? '₺';
        $minWithdrawal = $settings->min_bakiyecekim ?? 1000;
        $maxWithdrawal = $minWithdrawal * 50; // Minimum'un 50 katı

        return view('withdrawals.index', compact('withdrawals', 'user', 'currency', 'minWithdrawal', 'maxWithdrawal'));
    }

    // Çekim talebi oluştur (AJAX)
    public function store(Request $request)
    {
        // Settings'den dinamik değerleri al
        $settings = \App\Models\Setting::first();
        $minWithdrawal = $settings->min_bakiyecekim ?? 1000;
        $maxWithdrawal = $minWithdrawal * 50;

        $request->validate([
            'amount' => "required|numeric|min:{$minWithdrawal}|max:{$maxWithdrawal}",
            'iban' => 'required|string|size:26|regex:/^TR[0-9]{24}$/',
            'account_holder_name' => 'required|string|max:255'
        ]);

        $user = Auth::user();
        
        // Bakiye kontrolü
        if ($user->balance < $request->amount) {
            return response()->json([
                'success' => false,
                'message' => __('Insufficient balance.')
            ]);
        }

        // Minimum çekim tutarı kontrolü
        if ($request->amount < $minWithdrawal) {
            $currency = $settings->currency ?? '₺';
            return response()->json([
                'success' => false,
                'message' => __('Minimum amount is :amount :currency', ['amount' => number_format($minWithdrawal), 'currency' => $currency])
            ]);
        }

        // Bekleyen çekim talebi var mı?
        $pendingWithdrawal = WithdrawalRequest::where('user_id', $user->id)
            ->where('status', 'pending')
            ->exists();
            
        if ($pendingWithdrawal) {
            return response()->json([
                'success' => false,
                'message' => __('You already have a pending withdrawal request.')
            ]);
        }

        DB::beginTransaction();
        try {
            // Bakiyeyi dondur (çekim tutarını düş)
            $balanceBefore = $user->balance;
            $user->balance -= $request->amount;
            $user->save();

            // Çekim talebini oluştur
            $withdrawal = WithdrawalRequest::create([
                'user_id' => $user->id,
                'amount' => $request->amount,
                'iban' => strtoupper($request->iban),
                'account_holder_name' => $request->account_holder_name,
                'balance_before' => $balanceBefore,
                'balance_after' => $user->balance,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent()
            ]);

            // Balance log kaydı
            BalanceLog::create([
                'userId' => $user->id,
                'amount' => $request->amount,
                'type' => 'withdrawal_request',
                'description' => "Çekim talebi oluşturuldu - {$withdrawal->masked_iban}",
                'operation_type' => 'debit',
                'reference_type' => 'withdrawal',
                'reference_id' => $withdrawal->id,
                'balance_before' => $balanceBefore,
                'balance_after' => $user->balance
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => __('Withdrawal request created successfully. Your request will be processed within 1-3 business days.'),
                'withdrawal' => $withdrawal,
                'new_balance' => $user->balance
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while processing your request.'),
                'debug' => [
                    'error' => $e->getMessage(),
                    'line' => $e->getLine(),
                    'file' => $e->getFile()
                ]
            ]);
        }
    }

    // Çekim talebini iptal et
    public function cancel(Request $request, $id)
    {
        $user = Auth::user();
        $withdrawal = WithdrawalRequest::where('id', $id)
            ->where('user_id', $user->id)
            ->where('status', 'pending')
            ->first();

        if (!$withdrawal) {
            return response()->json([
                'success' => false,
                'message' => __('Withdrawal request not found or cannot be cancelled.')
            ]);
        }

        DB::beginTransaction();
        try {
            // Bakiyeyi geri ver
            $user->balance += $withdrawal->amount;
            $user->save();

            // Çekim talebini iptal et
            $withdrawal->status = 'rejected';
            $withdrawal->admin_note = 'Cancelled by user';
            $withdrawal->processed_at = now();
            $withdrawal->save();

            // Balance log kaydı
            BalanceLog::create([
                'user_id' => $user->id,
                'amount' => $withdrawal->amount,
                'type' => 'withdrawal_cancelled',
                'description' => "Çekim talebi iptal edildi - {$withdrawal->masked_iban}",
                'operation_type' => 'credit',
                'reference_type' => 'withdrawal',
                'reference_id' => $withdrawal->id,
                'balance_before' => $user->balance - $withdrawal->amount,
                'balance_after' => $user->balance
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => __('Withdrawal request cancelled successfully.'),
                'new_balance' => $user->balance
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while cancelling your request.')
            ]);
        }
    }

    // Admin: Çekim talepleri listesi
    public function adminIndex()
    {
        return view('admin.withdrawals.index');
    }

    // Admin: Çekim talepleri DataTable
    public function adminDataTable(Request $request)
    {
        // İstatistik verilerini döndür
        if ($request->has('get_stats') && $request->get_stats) {
            $stats = [
                'pending' => WithdrawalRequest::where('status', 'pending')->count(),
                'approved' => WithdrawalRequest::where('status', 'approved')->count(),
                'completed' => WithdrawalRequest::where('status', 'completed')->count(),
                'rejected' => WithdrawalRequest::where('status', 'rejected')->count(),
                'total_amount' => WithdrawalRequest::whereIn('status', ['pending', 'approved'])->sum('amount'),
                'paid_amount' => WithdrawalRequest::where('status', 'completed')->sum('amount')
            ];
            
            return response()->json(['stats' => $stats]);
        }

        $query = WithdrawalRequest::with(['user', 'processedBy']);

        // Search functionality
        if ($request->has('search') && $request->search['value']) {
            $searchValue = $request->search['value'];
            $query->whereHas('user', function($q) use ($searchValue) {
                $q->where('account_holder', 'like', "%{$searchValue}%")
                  ->orWhere('email', 'like', "%{$searchValue}%");
            })->orWhere('iban', 'like', "%{$searchValue}%")
              ->orWhere('amount', 'like', "%{$searchValue}%");
        }

        // Status filter
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        // Total records
        $totalRecords = $query->count();

        // Pagination
        $start = $request->start ?? 0;
        $length = $request->length ?? 10;
        
        $withdrawals = $query->skip($start)->take($length)
            ->orderBy('created_at', 'desc')
            ->get();

        $data = [];
        foreach ($withdrawals as $withdrawal) {
            $data[] = [
                'id' => $withdrawal->id,
                'user_name' => $withdrawal->user->account_holder ?? $withdrawal->user->name ?? 'N/A',
                'user_email' => $withdrawal->user->email,
                'amount' => $withdrawal->amount,
                'iban' => $withdrawal->iban, // Full IBAN for copying
                'iban_masked' => $withdrawal->masked_iban, // Masked for display
                'account_holder_name' => $withdrawal->account_holder_name,
                'status' => $withdrawal->status,
                'status_badge' => $withdrawal->status_badge,
                'status_text' => $withdrawal->status_text,
                'created_at' => $withdrawal->created_at->format('d.m.Y H:i'),
                'processed_by' => $withdrawal->processedBy->account_holder ?? $withdrawal->processedBy->name ?? '-',
                'admin_note' => $withdrawal->admin_note ?? '-',
                'actions' => $this->getAdminActionButtons($withdrawal)
            ];
        }

        return response()->json([
            'draw' => $request->draw,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $totalRecords,
            'data' => $data
        ]);
    }

    private function getAdminActionButtons($withdrawal)
    {
        $buttons = '<button class="btn btn-sm btn-info view-withdrawal" data-id="'.$withdrawal->id.'">
                        <i class="bx bx-show"></i> '.__('View').'
                    </button>';
        
        if ($withdrawal->status === 'pending') {
            $buttons .= ' <button class="btn btn-sm btn-success approve-withdrawal" data-id="'.$withdrawal->id.'">
                            <i class="bx bx-check"></i> '.__('Approve').'
                        </button>
                        <button class="btn btn-sm btn-danger reject-withdrawal" data-id="'.$withdrawal->id.'">
                            <i class="bx bx-x"></i> '.__('Reject').'
                        </button>';
        } elseif ($withdrawal->status === 'approved') {
            $buttons .= ' <button class="btn btn-sm btn-primary complete-withdrawal" data-id="'.$withdrawal->id.'">
                            <i class="bx bx-check-double"></i> '.__('Complete').'
                        </button>';
        }
        
        return $buttons;
    }

    // Admin: Çekim talebini onayla
    public function approve(Request $request, $id)
    {
        $withdrawal = WithdrawalRequest::findOrFail($id);
        
        if ($withdrawal->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => __('This withdrawal request cannot be approved.')
            ]);
        }

        $withdrawal->status = 'approved';
        $withdrawal->processed_by = Auth::id();
        $withdrawal->processed_at = now();
        $withdrawal->admin_note = $request->note ?? 'Approved by admin';
        $withdrawal->save();

        return response()->json([
            'success' => true,
            'message' => __('Withdrawal request approved successfully.')
        ]);
    }

    // Admin: Çekim talebini reddet
    public function reject(Request $request, $id)
    {
        $withdrawal = WithdrawalRequest::findOrFail($id);
        
        if ($withdrawal->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => __('This withdrawal request cannot be rejected.')
            ]);
        }

        DB::beginTransaction();
        try {
            // Bakiyeyi geri ver
            $user = $withdrawal->user;
            $user->balance += $withdrawal->amount;
            $user->save();

            // Çekim talebini reddet
            $withdrawal->status = 'rejected';
            $withdrawal->processed_by = Auth::id();
            $withdrawal->processed_at = now();
            $withdrawal->admin_note = $request->reason ?? 'Rejected by admin';
            $withdrawal->save();

            // Balance log kaydı
            BalanceLog::create([
                'user_id' => $user->id,
                'amount' => $withdrawal->amount,
                'type' => 'withdrawal_rejected',
                'description' => "Çekim talebi reddedildi - {$withdrawal->masked_iban}",
                'operation_type' => 'credit',
                'reference_type' => 'withdrawal',
                'reference_id' => $withdrawal->id,
                'balance_before' => $user->balance - $withdrawal->amount,
                'balance_after' => $user->balance
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => __('Withdrawal request rejected successfully.')
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => __('An error occurred while rejecting the request.')
            ]);
        }
    }

    // Admin: Çekim talebini tamamla
    public function complete(Request $request, $id)
    {
        $request->validate([
            'transaction_reference' => 'required|string|max:255'
        ]);

        $withdrawal = WithdrawalRequest::findOrFail($id);
        
        if ($withdrawal->status !== 'approved') {
            return response()->json([
                'success' => false,
                'message' => __('This withdrawal request cannot be completed.')
            ]);
        }

        $withdrawal->status = 'completed';
        $withdrawal->transaction_reference = $request->transaction_reference;
        $withdrawal->processed_by = Auth::id();
        $withdrawal->processed_at = now();
        $withdrawal->save();

        // Balance log kaydı
        BalanceLog::create([
            'user_id' => $withdrawal->user_id,
            'amount' => $withdrawal->amount,
            'type' => 'withdrawal_completed',
            'description' => "Çekim tamamlandı - {$withdrawal->masked_iban} - Ref: {$request->transaction_reference}",
            'operation_type' => 'debit',
            'reference_type' => 'withdrawal',
            'reference_id' => $withdrawal->id,
            'balance_before' => $withdrawal->user->balance,
            'balance_after' => $withdrawal->user->balance
        ]);

        return response()->json([
            'success' => true,
            'message' => __('Withdrawal completed successfully.')
        ]);
    }

    // Admin: Çekim talebi detayları
    public function details(Request $request, $id)
    {
        $withdrawal = WithdrawalRequest::with(['user', 'processedBy'])->findOrFail($id);
        
        $data = [
            'id' => $withdrawal->id,
            'user_name' => $withdrawal->user->account_holder ?? $withdrawal->user->name ?? 'N/A',
            'user_email' => $withdrawal->user->email,
            'user_phone' => $withdrawal->user->phone ?? null,
            'user_balance' => $withdrawal->user->balance,
            'amount' => $withdrawal->amount,
            'iban' => $withdrawal->iban,
            'account_holder_name' => $withdrawal->account_holder_name,
            'status' => $withdrawal->status,
            'status_text' => $withdrawal->status_text,
            'status_badge' => $withdrawal->status_badge,
            'balance_before' => $withdrawal->balance_before,
            'balance_after' => $withdrawal->balance_after,
            'admin_note' => $withdrawal->admin_note,
            'transaction_reference' => $withdrawal->transaction_reference,
            'processed_by' => $withdrawal->processedBy->account_holder ?? $withdrawal->processedBy->name ?? null,
            'processed_at' => $withdrawal->processed_at ? $withdrawal->processed_at->format('d.m.Y H:i') : null,
            'created_at' => $withdrawal->created_at->format('d.m.Y H:i'),
            'ip_address' => $withdrawal->ip_address,
            'user_agent' => $withdrawal->user_agent
        ];

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }
}