<?php

namespace App\Http\Helpers;

use App\Models\Setting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Http\Client\RequestException;

class RecaptchaHelper
{
    /**
     * reCAPTCHA doğrulaması yap
     */
    public static function verify($recaptchaResponse)
    {
        try {
            $settings = Setting::first();
            
            // reCAPTCHA aktif değilse doğrulama geç
            if (!$settings || !$settings->recaptcha_enabled) {
                return true;
            }
            
            // Secret key yoksa hata
            if (empty($settings->recaptcha_secret_key)) {
                Log::warning('reCAPTCHA secret key not found');
                return false;
            }
            
            // reCAPTCHA response yoksa hata
            if (empty($recaptchaResponse)) {
                return false;
            }
            
            // Direkt cURL kullan (HTTP Client sorunları nedeniyle)
            Log::info('Using cURL for reCAPTCHA verification (HTTP Client disabled due to server issues)');
            return self::verifyCurl($settings->recaptcha_secret_key, $recaptchaResponse);
            
        } catch (ConnectionException $e) {
            Log::error('reCAPTCHA connection error: ' . $e->getMessage());
            Log::info('Falling back to cURL for reCAPTCHA verification');
            return self::verifyCurl($settings->recaptcha_secret_key, $recaptchaResponse);
        } catch (RequestException $e) {
            Log::error('reCAPTCHA request error: ' . $e->getMessage());
            Log::info('Falling back to cURL due to request error');
            return self::verifyCurl($settings->recaptcha_secret_key, $recaptchaResponse);
        } catch (\Error $e) {
            Log::error('reCAPTCHA PHP error: ' . $e->getMessage());
            Log::info('Falling back to cURL due to PHP error');
            return self::verifyCurl($settings->recaptcha_secret_key, $recaptchaResponse);
        } catch (\Exception $e) {
            Log::error('reCAPTCHA verification error: ' . $e->getMessage());
            Log::info('Falling back to cURL due to general error');
            return self::verifyCurl($settings->recaptcha_secret_key, $recaptchaResponse);
        } catch (\Throwable $e) {
            Log::error('reCAPTCHA throwable error: ' . $e->getMessage());
            Log::info('Falling back to cURL due to throwable error');
            return self::verifyCurl($settings->recaptcha_secret_key, $recaptchaResponse);
        }
    }
    
    /**
     * cURL ile reCAPTCHA doğrulaması
     */
    private static function verifyCurl($secretKey, $recaptchaResponse)
    {
        try {
            $postData = http_build_query([
                'secret' => $secretKey,
                'response' => $recaptchaResponse,
                'remoteip' => request()->ip()
            ]);
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'https://www.google.com/recaptcha/api/siteverify');
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; reCAPTCHA-PHP)');
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                Log::error('cURL error in reCAPTCHA verification: ' . $error);
                return false;
            }
            
            if ($httpCode !== 200) {
                Log::error('HTTP error in reCAPTCHA verification: ' . $httpCode);
                return false;
            }
            
            $result = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::error('JSON decode error in reCAPTCHA verification: ' . json_last_error_msg());
                return false;
            }
            
            Log::info('reCAPTCHA verification result (cURL)', [
                'success' => $result['success'] ?? false,
                'error_codes' => $result['error-codes'] ?? [],
                'ip' => request()->ip(),
                'response_length' => strlen($response)
            ]);
            
            return $result['success'] ?? false;
            
        } catch (\Exception $e) {
            Log::error('cURL reCAPTCHA verification error: ' . $e->getMessage());
            return false;
        }
    }
}