<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class MultiSiteMiddleware
{
    public function handle(Request $request, Closure $next)
    {
        // Giriş/çıkış ekranlarını engelleme
        if ($request->routeIs([
            'site2.login', 'site2.login.submit',
            'site2.register', 'site2.landing',
            'login', 'logout',
        ])) {
            return $next($request);
        }

        // Host -> normalize
        $host = strtolower($request->getHost());
        $host = preg_replace('/^www\./', '', $host);
        $host = preg_replace('/:\d+$/', '', $host);

        // Site anahtarını çöz
        $site = $this->resolveSiteKeyFromHost($host);

        Log::debug('MultiSite incoming', [
            'host' => $host,
            'site' => $site,
            'route' => optional($request->route())->getName(),
            'path'  => $request->path(),
            'available_keys' => array_keys(config('multisite.sites') ?: []),
        ]);

        $strict = filter_var(env('MULTISITE_STRICT', true), FILTER_VALIDATE_BOOL);
        if (!$site) {
            if ($strict) {
                abort(403, 'Site is not configured.');
            }
            // debug modunda (MULTISITE_STRICT=false) geç
            return $next($request);
        }

        // İstersen global paylaş
        app()->instance('current_site', $site);
        view()->share('currentSite', $site);

        // *** DİKKAT: Burada ÜYELİK/ROL KONTROLÜ YOK! ***
        return $next($request);
    }

    protected function resolveSiteKeyFromHost(string $host): ?string
    {
        $host = strtolower(preg_replace('/^www\./', '', $host));
        $host = preg_replace('/:\d+$/', '', $host);

        $sites = config('multisite.sites') ?: [];
        if (empty($sites)) return null;

        if (array_key_exists($host, $sites)) return $host;

        foreach (array_keys($sites) as $key) {
            $keyNorm = strtolower(preg_replace('/^www\./', '', $key));
            if ($host === $keyNorm || str_ends_with($host, '.' . $keyNorm)) {
                return $key;
            }
        }

        foreach ($sites as $key => $conf) {
            $domains = array_map(function ($d) {
                $d = strtolower($d);
                $d = preg_replace('/^www\./', '', $d);
                return preg_replace('/:\d+$/', '', $d);
            }, (array)($conf['domains'] ?? []));

            foreach ($domains as $d) {
                if ($host === $d || str_ends_with($host, '.' . $d)) {
                    return $key;
                }
            }
        }

        foreach (array_filter([env('APP_URL'), env('SECONDARY_APP_URL')]) as $url) {
            $h = parse_url($url, PHP_URL_HOST);
            if (!$h) continue;
            $h = strtolower(preg_replace('/^www\./', '', $h));
            if ($host === $h || str_ends_with($host, '.' . $h)) {
                if (array_key_exists($h, $sites)) return $h;
            }
        }

        return null;
    }
}
