<?php

namespace App\Http\Controllers;

use App\Models\UserSession;
use Illuminate\Support\Facades\Auth;
use App\Models\SocialPlatform;
use Illuminate\Support\Facades\Cache;
use App\Models\Task;
use App\Models\User;
use App\Models\Country;
use App\Models\City;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TaskController extends Controller
{
    // Display a listing of the resource.
    public function index()
    {
        $tasks = Task::with(['proofs' => function($query) {
            $query->select('TaskId', DB::raw('count(*) as total'))
                  ->where('status', 1)
                  ->groupBy('TaskId');
        }])
        ->leftJoin('socialplatforms', 'tasks.category_id', '=', 'socialplatforms.id')
        ->leftJoin('users', 'tasks.user_id', '=', 'users.id')
        ->leftJoin('cities', 'tasks.city', '=', 'cities.id')
        ->leftJoin('countries', 'tasks.country_id', '=', 'countries.code')
        ->select(
            'tasks.*', 
            'socialplatforms.platform as PlatformName',
            'socialplatforms.icon as PlatformIcon',
            'socialplatforms.description as PlatformDescription',
            'users.account_holder as UserName',
            'users.email as UserEmail',
            'users.user_avatar as UserAvatar',
            'cities.name as CityName',
            'countries.name as CountryName'
        )
        ->get();
    
           
        $user = Auth::user();
        $platforms = SocialPlatform::where('status', 1)->get();
        $countries = Country::all();
        $cities = City::all();
        return view('tasks.index', compact('tasks', 'user', 'platforms','countries','cities'));
    }

    // Show the form for creating a new resource.
    public function create()
    {
        return view('tasks.create');
    }

    // Store a newly created resource in storage.
    public function store(Request $request)
    {
        $messages = [
            'name.required' => __('Task name is required.'),
            'link.required' => __('Task link must be a valid URL.'),
            'link.url' => __('Task link must be a valid URL.'),
            'description.required' => __('Task description is required.'),
            'category_id.required' => __('Task category is required.'),
            'image.required' => __('Task image is required.'),
            'image.image' => __('Task image must be an image file.'),
            'image.mimes' => __('Task image must be a file of type: jpeg, png, jpg, gif, svg.'),
            'image.max' => __('Task image must not be greater than 2MB.'),
            'reward.required' => __('Task reward is required.'),
            'limit.required' => __('Task limit is required.'),
            'user_limit.required' => __('Task user limit is required.'),
            'country_id.required' => __('Task country is required.'),
            'city.required' => __('Task city is required.'),
            'gender.required' => __('Task gender is required.'),
            'button_text.required' => __('Task button text is required.'),
            'proof_number.required' => __('Task proof number is required.'),
            'daily_limit.required' => __('Task daily limit is required.'),
            'daily_limit.integer' => __('Task daily limit must be an integer.'),
            'daily_limit.numeric' => __('Task daily limit must be a number.'),
            

            // Add other messages as necessary
        ];
    
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'link' => 'required|url',
            'description' => 'required|string|max:1000',
            'category_id' => 'required|integer',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'reward' => 'required|numeric',
            'limit' => 'required|integer',
            'user_limit' => 'required|integer',
            'country_id' => 'required|integer',
            'city' => 'integer',
            'gender' => 'required|integer',
            'button_text' => 'required|string|max:500',
            'proof_number' => 'required|integer',
            'daily_limit' => 'nullable|integer',
          
        ], $messages);
    
        $task = new Task($validated);
        $task->user_id = Auth::id(); 
        $task->status = '2'; 
        $task->gcode = "0";
        $task->pin = "0";
        $task->type = "1";
    
        if ($request->hasFile('image')) {
            $imageName = time().'.'.$request->image->extension();  
            $request->image->move(public_path('assets/images/tasks'), $imageName);
            $task->image = $imageName;
        }
    
        $task->save();
    
        return redirect()->route('tasks.index')->with('success', 'Task created successfully');
    }
    

    public function getCitiesByCountry($countryId)
{
    $cities = City::where('country_id', $countryId)->get();
    return response()->json($cities);
}

    // Display the specified resource.
    public function show(Task $task)
    {
        return view('tasks.show', compact('task'));
    }

    // Show the form for editing the specified resource.
    public function edit(Task $task)
    {
        return view('tasks.edit', compact('task'));
    }

    // Update the specified resource in storage.
    public function update(Request $request, Task $task)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            // Validate other fields as necessary
        ]);

        $task->update($validated);
        return redirect()->route('tasks.index');
    }

    // Remove the specified resource from storage.
    public function destroy(Task $task)
    {
        $task->delete();
        return redirect()->route('tasks.index');
    }
}
