<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Announcement extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'content',
        'type',
        'priority',
        'is_active',
        'show_as_popup',
        'expires_at',
        'created_by'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'show_as_popup' => 'boolean',
        'expires_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relations
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function targetUsers()
    {
        return $this->belongsToMany(User::class, 'announcement_users');
    }

    public function readByUsers()
    {
        return $this->hasMany(AnnouncementUser::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                    ->where(function($q) {
                        $q->whereNull('expires_at')
                          ->orWhere('expires_at', '>', now());
                    });
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where(function($q) use ($userId) {
            $q->where('type', 'general')
              ->orWhereHas('targetUsers', function($userQuery) use ($userId) {
                  $userQuery->where('user_id', $userId);
              });
        });
    }

    public function scopeUnreadByUser($query, $userId)
    {
        return $query->whereDoesntHave('readByUsers', function($readQuery) use ($userId) {
            $readQuery->where('user_id', $userId);
        });
    }

    // Accessors
    public function getPriorityBadgeAttribute()
    {
        $badges = [
            'low' => 'success',
            'medium' => 'info',
            'high' => 'warning',
            'urgent' => 'danger'
        ];
        
        return $badges[$this->priority] ?? 'secondary';
    }

    public function getPriorityTextAttribute()
    {
        $texts = [
            'low' => __('Low'),
            'medium' => __('Medium'),
            'high' => __('High'),
            'urgent' => __('Urgent')
        ];
        
        return $texts[$this->priority] ?? $this->priority;
    }

    public function getTypeTextAttribute()
    {
        $texts = [
            'general' => __('General'),
            'user_specific' => __('User Specific')
        ];
        
        return $texts[$this->type] ?? $this->type;
    }

    public function getFormattedContentAttribute()
    {
        return nl2br(e($this->content));
    }

    // Methods
    public function isExpired()
    {
        return $this->expires_at && $this->expires_at->isPast();
    }

    public function isReadByUser($userId)
    {
        return $this->readByUsers()->where('user_id', $userId)->exists();
    }

    public function markAsReadByUser($userId)
    {
        if (!$this->isReadByUser($userId)) {
            AnnouncementUser::create([
                'announcement_id' => $this->id,
                'user_id' => $userId
            ]);
        }
    }

    public function canBeSeenByUser($userId)
    {
        if (!$this->is_active || $this->isExpired()) {
            return false;
        }

        if ($this->type === 'general') {
            return true;
        }

        return $this->targetUsers()->where('user_id', $userId)->exists();
    }
}