<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Backup extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'original_name',
        'file_path',
        'file_size',
        'type',
        'status',
        'progress',
        'status_message',
        'description',
        'created_by',
        'download_count',
        'checksum'
    ];

    protected $casts = [
        'file_size' => 'integer',
        'progress' => 'integer',
        'download_count' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relations
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scopes
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeRecent($query, $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    // Accessors
    public function getFormattedSizeAttribute()
    {
        return $this->formatBytes($this->file_size);
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'pending' => 'warning',
            'processing' => 'info',
            'completed' => 'success',
            'failed' => 'danger',
            'cancelled' => 'secondary'
        ];
        
        return $badges[$this->status] ?? 'secondary';
    }

    public function getStatusTextAttribute()
    {
        $texts = [
            'pending' => __('Pending'),
            'processing' => __('Processing'),
            'completed' => __('Completed'),
            'failed' => __('Failed'),
            'cancelled' => __('Cancelled')
        ];
        
        return $texts[$this->status] ?? $this->status;
    }

    public function getTypeTextAttribute()
    {
        $types = [
            'full' => __('Full Backup'),
            'database' => __('Database Only'),
            'files' => __('Files Only')
        ];
        
        return $types[$this->type] ?? $this->type;
    }

    public function getTypeBadgeAttribute()
    {
        $badges = [
            'full' => 'primary',
            'database' => 'success',
            'files' => 'info'
        ];
        
        return $badges[$this->type] ?? 'secondary';
    }

    public function getAgeAttribute()
    {
        return $this->created_at->diffForHumans();
    }

    public function getIsRestorable()
    {
        return $this->status === 'completed' && $this->type === 'database';
    }

    // Methods
    public function canBeDownloaded()
    {
        if ($this->status !== 'completed' || !$this->file_path) {
            return false;
        }
        
        try {
            // Ensure backup disk exists before checking
            $this->ensureBackupDiskExists();
            return \Storage::disk('backups')->exists($this->file_path);
        } catch (\Exception $e) {
            return false;
        }
    }

    public function canBeRestored()
    {
        if ($this->status !== 'completed' || $this->type !== 'database' || !$this->file_path) {
            return false;
        }
        
        try {
            // Ensure backup disk exists before checking
            $this->ensureBackupDiskExists();
            return \Storage::disk('backups')->exists($this->file_path);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Ensure backup disk configuration exists
     */
    protected function ensureBackupDiskExists()
    {
        if (!config('filesystems.disks.backups')) {
            $backupPath = storage_path('app/backups');
            if (!file_exists($backupPath)) {
                mkdir($backupPath, 0755, true);
            }

            config([
                'filesystems.disks.backups' => [
                    'driver' => 'local',
                    'root' => $backupPath,
                    'url' => env('APP_URL').'/storage/backups',
                    'visibility' => 'private',
                ]
            ]);
        }
    }

    public function updateProgress($progress, $message = null)
    {
        $this->update([
            'progress' => $progress,
            'status_message' => $message,
            'status' => $progress >= 100 ? 'completed' : 'processing'
        ]);
    }

    public function markAsCompleted($fileSize = null, $checksum = null)
    {
        $this->update([
            'status' => 'completed',
            'progress' => 100,
            'file_size' => $fileSize,
            'checksum' => $checksum,
            'status_message' => __('Backup completed successfully')
        ]);
    }

    public function markAsFailed($message = null)
    {
        $this->update([
            'status' => 'failed',
            'status_message' => $message ?? __('Backup failed')
        ]);
    }

    private function formatBytes($size, $precision = 2)
    {
        if ($size === 0) return '0 B';
        
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $base = log($size, 1024);
        
        return round(pow(1024, $base - floor($base)), $precision) . ' ' . $units[floor($base)];
    }
}