<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Language extends Model
{
    protected $fillable = [
        'code',
        'name',
        'native_name',
        'flag',
        'is_active',
        'is_default',
        'is_rtl',
        'sort_order'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'is_default' => 'boolean',
        'is_rtl' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get all translations for this language
     */
    public function translations(): HasMany
    {
        return $this->hasMany(LanguageTranslation::class);
    }

    /**
     * Get approved translations for this language
     */
    public function approvedTranslations(): HasMany
    {
        return $this->hasMany(LanguageTranslation::class)->where('is_approved', true);
    }

    /**
     * Scope for active languages
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for ordered languages
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('name');
    }

    /**
     * Get the default language
     */
    public static function getDefault()
    {
        return static::where('is_default', true)->first();
    }

    /**
     * Set this language as default
     */
    public function setAsDefault()
    {
        // Remove default from all languages
        static::where('is_default', true)->update(['is_default' => false]);
        
        // Set this as default
        $this->update(['is_default' => true]);
    }

    /**
     * Get translation count for this language
     */
    public function getTranslationCountAttribute()
    {
        return $this->translations()->count();
    }

    /**
     * Get approved translation count for this language
     */
    public function getApprovedTranslationCountAttribute()
    {
        return $this->approvedTranslations()->count();
    }

    /**
     * Get completion percentage
     */
    public function getCompletionPercentageAttribute()
    {
        $totalKeys = LanguageKey::count();
        if ($totalKeys === 0) return 0;
        
        $translatedKeys = $this->approvedTranslations()->count();
        return round(($translatedKeys / $totalKeys) * 100, 2);
    }

    /**
     * Export translations to JSON format
     */
    public function exportToJson()
    {
        $translations = [];
        
        $this->approvedTranslations()
            ->with('languageKey')
            ->get()
            ->each(function ($translation) use (&$translations) {
                $translations[$translation->languageKey->key] = $translation->value;
            });
            
        return $translations;
    }

    /**
     * Get flag emoji or default
     */
    public function getFlagDisplayAttribute()
    {
        return $this->flag ?: '🌐';
    }
}