<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class LanguageKey extends Model
{
    protected $fillable = [
        'key',
        'description',
        'group'
    ];

    /**
     * Get all translations for this key
     */
    public function translations(): HasMany
    {
        return $this->hasMany(LanguageTranslation::class);
    }

    /**
     * Get approved translations for this key
     */
    public function approvedTranslations(): HasMany
    {
        return $this->hasMany(LanguageTranslation::class)->where('is_approved', true);
    }

    /**
     * Get translation for specific language
     */
    public function getTranslationForLanguage($languageId)
    {
        return $this->translations()->where('language_id', $languageId)->first();
    }

    /**
     * Get approved translation for specific language
     */
    public function getApprovedTranslationForLanguage($languageId)
    {
        return $this->approvedTranslations()->where('language_id', $languageId)->first();
    }

    /**
     * Scope for specific group
     */
    public function scopeGroup($query, $group)
    {
        return $query->where('group', $group);
    }

    /**
     * Scope for search
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('key', 'LIKE', "%{$search}%")
              ->orWhere('description', 'LIKE', "%{$search}%");
        });
    }

    /**
     * Get all available groups
     */
    public static function getGroups()
    {
        return static::distinct('group')->pluck('group')->sort()->values();
    }

    /**
     * Get translation count for this key
     */
    public function getTranslationCountAttribute()
    {
        return $this->translations()->count();
    }

    /**
     * Get approved translation count for this key
     */
    public function getApprovedTranslationCountAttribute()
    {
        return $this->approvedTranslations()->count();
    }

    /**
     * Check if key has translation for all active languages
     */
    public function getIsCompleteAttribute()
    {
        $activeLanguageCount = Language::active()->count();
        return $this->approved_translation_count >= $activeLanguageCount;
    }

    /**
     * Get missing languages for this key
     */
    public function getMissingLanguages()
    {
        $translatedLanguageIds = $this->approvedTranslations()->pluck('language_id');
        
        return Language::active()
            ->whereNotIn('id', $translatedLanguageIds)
            ->get();
    }

    /**
     * Import from existing JSON files
     */
    public static function importFromJson($filePath, $group = 'general')
    {
        if (!file_exists($filePath)) {
            return false;
        }

        $data = json_decode(file_get_contents($filePath), true);
        if (!$data) {
            return false;
        }

        $imported = 0;
        foreach ($data as $key => $value) {
            $languageKey = static::firstOrCreate([
                'key' => $key,
                'group' => $group
            ]);
            
            if ($languageKey->wasRecentlyCreated) {
                $imported++;
            }
        }

        return $imported;
    }
}