<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class LanguageTranslation extends Model
{
    protected $fillable = [
        'language_id',
        'language_key_id',
        'value',
        'is_approved',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'is_approved' => 'boolean',
        'language_id' => 'integer',
        'language_key_id' => 'integer',
        'created_by' => 'integer',
        'updated_by' => 'integer',
    ];

    /**
     * Get the language this translation belongs to
     */
    public function language(): BelongsTo
    {
        return $this->belongsTo(Language::class);
    }

    /**
     * Get the language key this translation belongs to
     */
    public function languageKey(): BelongsTo
    {
        return $this->belongsTo(LanguageKey::class);
    }

    /**
     * Get the user who created this translation
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who last updated this translation
     */
    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Scope for approved translations
     */
    public function scopeApproved($query)
    {
        return $query->where('is_approved', true);
    }

    /**
     * Scope for pending translations
     */
    public function scopePending($query)
    {
        return $query->where('is_approved', false);
    }

    /**
     * Scope for specific language
     */
    public function scopeForLanguage($query, $languageId)
    {
        return $query->where('language_id', $languageId);
    }

    /**
     * Scope for specific group
     */
    public function scopeForGroup($query, $group)
    {
        return $query->whereHas('languageKey', function ($q) use ($group) {
            $q->where('group', $group);
        });
    }

    /**
     * Scope for search
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('value', 'LIKE', "%{$search}%")
              ->orWhereHas('languageKey', function ($keyQuery) use ($search) {
                  $keyQuery->where('key', 'LIKE', "%{$search}%")
                           ->orWhere('description', 'LIKE', "%{$search}%");
              });
        });
    }

    /**
     * Approve this translation
     */
    public function approve($userId = null)
    {
        $this->update([
            'is_approved' => true,
            'updated_by' => $userId
        ]);
    }

    /**
     * Reject this translation
     */
    public function reject($userId = null)
    {
        $this->update([
            'is_approved' => false,
            'updated_by' => $userId
        ]);
    }

    /**
     * Create or update translation
     */
    public static function createOrUpdate($languageId, $languageKeyId, $value, $userId = null, $autoApprove = false)
    {
        $translation = static::updateOrCreate(
            [
                'language_id' => $languageId,
                'language_key_id' => $languageKeyId
            ],
            [
                'value' => $value,
                'is_approved' => $autoApprove,
                'updated_by' => $userId
            ]
        );

        if ($translation->wasRecentlyCreated) {
            $translation->update(['created_by' => $userId]);
        }

        return $translation;
    }

    /**
     * Get character count
     */
    public function getCharacterCountAttribute()
    {
        return mb_strlen($this->value);
    }

    /**
     * Get word count
     */
    public function getWordCountAttribute()
    {
        return str_word_count(strip_tags($this->value));
    }

    /**
     * Check if translation contains HTML
     */
    public function getHasHtmlAttribute()
    {
        return $this->value !== strip_tags($this->value);
    }

    /**
     * Get status badge class
     */
    public function getStatusBadgeClassAttribute()
    {
        return $this->is_approved ? 'bg-success' : 'bg-warning';
    }

    /**
     * Get status text
     */
    public function getStatusTextAttribute()
    {
        return $this->is_approved ? __('Approved') : __('Pending');
    }
}