<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Order extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_number',
        'user_id',
        'product_id',
        'product_name',
        'product_price',
        'quantity',
        'total_amount',
        'status',
        'payment_status',
        'product_data',
        'delivery_data',
        'delivered_at',
        'notes'
    ];

    protected $casts = [
        'product_price' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'product_data' => 'array',
        'delivery_data' => 'array',
        'delivered_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relations
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopePaid($query)
    {
        return $query->where('payment_status', 'paid');
    }

    // Accessors
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'pending' => 'warning',
            'processing' => 'info',
            'completed' => 'success',
            'cancelled' => 'secondary',
            'refunded' => 'danger'
        ];
        
        return $badges[$this->status] ?? 'secondary';
    }

    public function getStatusTextAttribute()
    {
        $texts = [
            'pending' => __('Pending'),
            'processing' => __('Processing'),
            'completed' => __('Completed'),
            'cancelled' => __('Cancelled'),
            'refunded' => __('Refunded')
        ];
        
        return $texts[$this->status] ?? $this->status;
    }

    public function getPaymentStatusBadgeAttribute()
    {
        $badges = [
            'pending' => 'warning',
            'paid' => 'success',
            'failed' => 'danger',
            'refunded' => 'secondary'
        ];
        
        return $badges[$this->payment_status] ?? 'secondary';
    }

    public function getPaymentStatusTextAttribute()
    {
        $texts = [
            'pending' => __('Pending'),
            'paid' => __('Paid'),
            'failed' => __('Failed'),
            'refunded' => __('Refunded')
        ];
        
        return $texts[$this->payment_status] ?? $this->payment_status;
    }

    // Methods
    public static function generateOrderNumber()
    {
        do {
            $orderNumber = 'ORD-' . date('Ymd') . '-' . strtoupper(substr(uniqid(), -6));
        } while (self::where('order_number', $orderNumber)->exists());
        
        return $orderNumber;
    }

    public function canBeCancelled()
    {
        return in_array($this->status, ['pending', 'processing']) && 
               $this->payment_status !== 'paid';
    }

    public function canBeRefunded()
    {
        return $this->status === 'completed' && 
               $this->payment_status === 'paid' &&
               $this->created_at->diffInDays(now()) <= 7; // 7 days refund policy
    }

    public function markAsDelivered($deliveryData = null)
    {
        $this->update([
            'status' => 'completed',
            'delivered_at' => now(),
            'delivery_data' => $deliveryData
        ]);
    }

    public function getFormattedTotalAttribute()
    {
        return number_format($this->total_amount, 2) . ' ₺';
    }

    public function getCanBeCancelledAttribute()
    {
        return $this->canBeCancelled();
    }
}