<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'product_name',
        'description',
        'price',
        'image_url',
        'category_id',
        'min_quantity',
        'max_quantity',
        'completing_time',
        'provider',
        'status',
        'is_digital',
        'stock_quantity',
        'digital_content',
        'auto_delivery',
        'delivery_instructions'
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'is_digital' => 'boolean',
        'auto_delivery' => 'boolean',
        'digital_content' => 'array',
        'status' => 'boolean'
    ];

    public function category()
    {
        return $this->belongsTo(ProductCategory::class, 'category_id');
    }

    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', true);
    }

    public function scopeDigital($query)
    {
        return $query->where('is_digital', true);
    }

    public function scopeInStock($query)
    {
        return $query->where(function($q) {
            $q->where('is_digital', true)
              ->orWhere('stock_quantity', '>', 0);
        });
    }

    // Accessors
    public function getFormattedPriceAttribute()
    {
        return number_format($this->price, 2) . ' ₺';
    }

    public function getStatusTextAttribute()
    {
        return $this->status ? __('Active') : __('Inactive');
    }

    public function getStatusBadgeAttribute()
    {
        return $this->status ? 'success' : 'secondary';
    }

    public function getStockStatusAttribute()
    {
        if ($this->is_digital) {
            return __('Digital Product');
        }
        
        if ($this->stock_quantity > 10) {
            return __('In Stock');
        } elseif ($this->stock_quantity > 0) {
            return __('Low Stock');
        } else {
            return __('Out of Stock');
        }
    }

    public function getStockBadgeAttribute()
    {
        if ($this->is_digital) {
            return 'info';
        }
        
        if ($this->stock_quantity > 10) {
            return 'success';
        } elseif ($this->stock_quantity > 0) {
            return 'warning';
        } else {
            return 'danger';
        }
    }

    // Methods
    public function isAvailable()
    {
        return $this->status && ($this->is_digital || $this->stock_quantity > 0);
    }

    public function canBePurchased($quantity = 1)
    {
        if (!$this->isAvailable()) {
            return false;
        }

        if ($quantity < $this->min_quantity || $quantity > $this->max_quantity) {
            return false;
        }

        if (!$this->is_digital && $this->stock_quantity < $quantity) {
            return false;
        }

        return true;
    }

    public function decreaseStock($quantity)
    {
        if (!$this->is_digital) {
            $this->decrement('stock_quantity', $quantity);
        }
    }

    public function increaseStock($quantity)
    {
        if (!$this->is_digital) {
            $this->increment('stock_quantity', $quantity);
        }
    }
}
