<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SocialMediaAccount extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id', 
        'platform_id', 
        'username', 
        'status',
        'approved_at',
        'rejected_at'
    ];

    protected $casts = [
        'approved_at' => 'datetime',
        'rejected_at' => 'datetime',
    ];

    // Status constants
    const STATUS_PENDING = 0;
    const STATUS_APPROVED = 1;
    const STATUS_REJECTED = 2;

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function platform()
    {
        return $this->belongsTo(SocialPlatform::class, 'platform_id');
    }

    // Scope for pending accounts
    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    // Scope for approved accounts
    public function scopeApproved($query)
    {
        return $query->where('status', self::STATUS_APPROVED);
    }

    // Scope for rejected accounts
    public function scopeRejected($query)
    {
        return $query->where('status', self::STATUS_REJECTED);
    }

    // Get status text
    public function getStatusTextAttribute()
    {
        switch ($this->status) {
            case self::STATUS_PENDING:
                return __('Pending');
            case self::STATUS_APPROVED:
                return __('Approved');
            case self::STATUS_REJECTED:
                return __('Rejected');
            default:
                return __('Unknown');
        }
    }

    // Check if account is approved
    public function isApproved()
    {
        return $this->status === self::STATUS_APPROVED;
    }

    // Check if account is pending
    public function isPending()
    {
        return $this->status === self::STATUS_PENDING;
    }

    // Check if account is rejected
    public function isRejected()
    {
        return $this->status === self::STATUS_REJECTED;
    }
}
