<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Ticket extends Model
{
    use HasFactory;

    protected $fillable = [
        'ticket_number',
        'user_id',
        'subject',
        'description',
        'priority',
        'category',
        'status',
        'assigned_to',
        'last_activity_at',
        'resolved_at',
        'closed_at',
        'attachments'
    ];

    protected $casts = [
        'attachments' => 'array',
        'last_activity_at' => 'datetime',
        'resolved_at' => 'datetime',
        'closed_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relations
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function assignedTo()
    {
        return $this->belongsTo(User::class, 'assigned_to');
    }

    public function messages()
    {
        return $this->hasMany(TicketMessage::class)->orderBy('created_at', 'asc');
    }

    public function latestMessage()
    {
        return $this->hasOne(TicketMessage::class)->latest();
    }

    public function unreadMessages()
    {
        return $this->hasMany(TicketMessage::class)->where('is_read', false);
    }

    // Accessors
    public function getPriorityBadgeAttribute()
    {
        $badges = [
            'low' => 'success',
            'medium' => 'warning',
            'high' => 'danger',
            'urgent' => 'dark'
        ];
        
        return $badges[$this->priority] ?? 'secondary';
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'open' => 'primary',
            'in_progress' => 'info',
            'waiting_user' => 'warning',
            'waiting_admin' => 'warning',
            'resolved' => 'success',
            'closed' => 'secondary'
        ];
        
        return $badges[$this->status] ?? 'secondary';
    }

    public function getPriorityTextAttribute()
    {
        $texts = [
            'low' => __('Low'),
            'medium' => __('Medium'),
            'high' => __('High'),
            'urgent' => __('Urgent')
        ];
        
        return $texts[$this->priority] ?? $this->priority;
    }

    public function getStatusTextAttribute()
    {
        $texts = [
            'open' => __('Open'),
            'in_progress' => __('In Progress'),
            'waiting_user' => __('Waiting for User'),
            'waiting_admin' => __('Waiting for Admin'),
            'resolved' => __('Resolved'),
            'closed' => __('Closed')
        ];
        
        return $texts[$this->status] ?? $this->status;
    }

    public function getCategoryTextAttribute()
    {
        $texts = [
            'general' => __('General'),
            'technical' => __('Technical'),
            'billing' => __('Billing'),
            'account' => __('Account'),
            'withdrawal' => __('Withdrawal'),
            'task' => __('Task')
        ];
        
        return $texts[$this->category] ?? $this->category;
    }

    // Scopes
    public function scopeOpen($query)
    {
        return $query->whereIn('status', ['open', 'in_progress', 'waiting_user', 'waiting_admin']);
    }

    public function scopeClosed($query)
    {
        return $query->whereIn('status', ['resolved', 'closed']);
    }

    public function scopeByPriority($query, $priority)
    {
        return $query->where('priority', $priority);
    }

    public function scopeByCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    public function scopeAssignedTo($query, $userId)
    {
        return $query->where('assigned_to', $userId);
    }

    public function scopeUnassigned($query)
    {
        return $query->whereNull('assigned_to');
    }

    // Methods
    public static function generateTicketNumber()
    {
        $year = date('Y');
        $lastTicket = self::where('ticket_number', 'like', "TKT-{$year}-%")
            ->orderBy('id', 'desc')
            ->first();
            
        if ($lastTicket) {
            $lastNumber = (int) substr($lastTicket->ticket_number, -3);
            $newNumber = str_pad($lastNumber + 1, 3, '0', STR_PAD_LEFT);
        } else {
            $newNumber = '001';
        }
        
        return "TKT-{$year}-{$newNumber}";
    }

    public function updateLastActivity()
    {
        $this->update(['last_activity_at' => now()]);
    }

    public function markAsResolved()
    {
        $this->update([
            'status' => 'resolved',
            'resolved_at' => now(),
            'last_activity_at' => now()
        ]);
    }

    public function markAsClosed()
    {
        $this->update([
            'status' => 'closed',
            'closed_at' => now(),
            'last_activity_at' => now()
        ]);
    }

    public function reopen()
    {
        $this->update([
            'status' => 'open',
            'resolved_at' => null,
            'closed_at' => null,
            'last_activity_at' => now()
        ]);
    }

    public function getUnreadCountForUser($userId)
    {
        return $this->messages()
            ->where('user_id', '!=', $userId)
            ->where('is_read', false)
            ->count();
    }
}