<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class TicketMessage extends Model
{
    use HasFactory;

    protected $fillable = [
        'ticket_id',
        'user_id',
        'message',
        'attachments',
        'is_internal',
        'is_read',
        'read_at'
    ];

    protected $casts = [
        'attachments' => 'array',
        'is_internal' => 'boolean',
        'is_read' => 'boolean',
        'read_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relations
    public function ticket()
    {
        return $this->belongsTo(Ticket::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    // Accessors
    public function getIsAdminAttribute()
    {
        return $this->user && $this->user->user_type == 1;
    }

    public function getFormattedMessageAttribute()
    {
        return nl2br(e($this->message));
    }

    // Methods
    public function markAsRead($userId = null)
    {
        if ($userId && $this->user_id == $userId) {
            return; // Don't mark own messages as read
        }

        $this->update([
            'is_read' => true,
            'read_at' => now()
        ]);
    }

    public function hasAttachments()
    {
        return !empty($this->attachments);
    }

    public function getAttachmentCount()
    {
        return $this->attachments ? count($this->attachments) : 0;
    }
}