<?php



namespace App\Models;



use Illuminate\Database\Eloquent\Factories\HasFactory;

use Illuminate\Foundation\Auth\User as Authenticatable;

use Illuminate\Notifications\Notifiable;

use Illuminate\Support\Facades\Cache;

use Illuminate\Support\Facades\Log;

use App\Helpers\UserHelper;



class User extends Authenticatable

{

    // User types

    const TYPE_ADMIN = 1;

    const TYPE_USER = 2;

    use HasFactory, Notifiable;



    protected $fillable = [

        'username',

        'email',

        'phone',

        'password',

        'sifre',

        'password_updated',

        'remember_token',

        'account_info',

        'account_holder',

        'balance',

        'iban',

        'bank_name',

        'referral_code',

        'verified',

        'membership_type',

        'membership_ends_at',

        'city_id',

        'country_id',

        'gender',

        'daily_bonus',

        'last_bonus_date',

        'frozen',

        'user_type',

        'is_online',

        'ip',

        'deleted',

        'banned',

        'ban_reason',

        'user_avatar',

    ];



    protected $hidden = [

        'password',

        'remember_token',

        'sifre',

    ];



    protected $casts = [

        'phone_verified' => 'boolean',

        'verified' => 'boolean',

        'frozen' => 'boolean',

        'is_online' => 'boolean',

        'password_updated' => 'boolean',

        'balance' => 'decimal:2',

        'last_bonus_date' => 'date',

      'membership_type' => 'string',

    ];



    // IBAN ilişkili metodlar

    public function getMaskedIbanAttribute()

    {

        if (!$this->iban || strlen($this->iban) <= 8) {

            return $this->iban;

        }

        return substr($this->iban, 0, 4) . str_repeat('*', strlen($this->iban) - 8) . substr($this->iban, -4);

    }



    public function hasCompletePaymentInfo()

    {

        return !empty($this->iban) && !empty($this->account_holder_name);

    }



    public function canRequestWithdrawal()

    {

        return $this->balance >= 1000 && $this->hasCompletePaymentInfo();

    }



    // Withdrawal requests ilişkisi

    public function withdrawalRequests()

    {

        return $this->hasMany(\App\Models\WithdrawalRequest::class);

    }



    public function tickets()

    {

        return $this->hasMany(\App\Models\Ticket::class);

    }



    public function assignedTickets()

    {

        return $this->hasMany(\App\Models\Ticket::class, 'assigned_to');

    }



    public function ticketMessages()

    {

        return $this->hasMany(\App\Models\TicketMessage::class);

    }



    public function pendingWithdrawals()

    {

        return $this->withdrawalRequests()->where('status', 'pending');

    }



    /**

     * Set the user's last activity time and online status in cache.

     */

    public static function boot()

    {

        parent::boot();

        

        // Kullanıcı güncellendiğinde cache'i temizle

        static::updating(function ($user) {

            if ($user->isDirty('is_online') && $user->is_online) {

                // Set or renew the cache for the user's online status.

                Cache::put('user-online-' . $user->id, true, now()->addMinutes(5));

            }

            // UserHelper cache'ini temizle

            UserHelper::clearUserCache($user->id);

        });

        

        // Kullanıcı kaydedildiğinde de cache'i temizle

        static::saved(function ($user) {

            UserHelper::clearUserCache($user->id);

        });

    }

    public function tasks()

    {

        return $this->belongsToMany(Task::class, 'task_user')

            ->using(TaskUser::class)

            ->withPivot('watched_time', 'is_completed')

            ->withTimestamps();

    }

    /**

     * Determine if the user is online based on cached data.

     */

    public function isOnline()

    {

        return Cache::has('user-online-' . $this->id);

    }



    public function balanceLogs()

    {

        return $this->hasMany(BalanceLog::class, 'userId');

    }



    public function socialMediaAccounts()

    {

        return $this->hasMany(SocialMediaAccount::class, 'user_id');

    }



    public function country()

    {

        return $this->belongsTo(Country::class, 'country_id', 'id');

    }



    public function city()

    {

        return $this->belongsTo(City::class, 'city_id');

    }



    public function taskProofs()

    {

        return $this->hasMany(TaskProof::class, 'UserId');

    }



    public function websiteVisitTasks()

    {

        return $this->hasMany(WebsiteVisitTask::class, 'created_by');

    }



    public function websiteVisits()

    {

        return $this->hasMany(WebsiteVisit::class, 'user_id');

    }



    // Helper methods for roles

    public function isAdmin()

    {

        return (int)$this->user_type === self::TYPE_ADMIN;

    }



    public function isUser()

    {

        return $this->user_type === self::TYPE_USER;

    }



    public function hasRole($role)

    {

        return $this->user_type === $role;

    }



    // Check if user can manage website visit tasks

    public function canManageWebsiteVisitTasks()

    {

        return $this->isAdmin();

    }



    // Check if user can view specific website visit task

    public function canViewWebsiteVisitTask(WebsiteVisitTask $task)

    {

        return $this->isAdmin() || $task->created_by === $this->id;

    }



    // Check if user can edit specific website visit task

    public function canEditWebsiteVisitTask(WebsiteVisitTask $task)

    {

        return $this->isAdmin() || $task->created_by === $this->id;

    }



    // Check if user can delete specific website visit task

    public function canDeleteWebsiteVisitTask(WebsiteVisitTask $task)

    {

        return $this->isAdmin(); // Only admin can delete

    }



    /**

     * Günlük bonus kontrolü ve ekleme

     */

    public function checkAndAddDailyBonus()

    {

        try {

            // Settings'ten günlük bonus miktarını al

            $settings = \App\Models\Setting::first();

            if (!$settings) {

                Log::warning('Settings not found for daily bonus check');

                return false;

            }

            

            $dailyBonusAmount = $settings->gunluk_bonus ?? 0;

            

            // Eğer günlük bonus 0 ise işlem yapma

            if ($dailyBonusAmount <= 0) {

                return false;

            }

            

            // Bugünün tarihi

            $today = now()->toDateString();

            

            // Eğer bugün bonus almamışsa (tarih karşılaştırması)

            if ($this->last_bonus_date === null || $this->last_bonus_date->toDateString() !== $today) {

                // Mevcut bakiyeyi kaydet

                $oldBalance = $this->balance;

                

                // Bakiyeye bonus ekle

                $this->balance += $dailyBonusAmount;

                $this->last_bonus_date = now()->toDateString(); // Bugünün tarihini kaydet

                $this->save();

                

                // Balance log kaydı oluştur

                \App\Models\BalanceLog::create([

                    'userId' => $this->id,

                    'balance_before' => $oldBalance,

                    'balance' => $dailyBonusAmount,

                    'balance_after' => $this->balance,

                    'description' => 'Günlük Bonus',

                    'type' => 'daily_bonus',

                    'created_at' => now(),

                    'updated_at' => now()

                ]);

                

                Log::info('Daily bonus added for user ' . $this->id . ': ' . $dailyBonusAmount);

                

                return [

                    'bonus_added' => true,

                    'amount' => $dailyBonusAmount,

                    'new_balance' => $this->balance

                ];

            }

            

            return false;

            

        } catch (\Exception $e) {

            Log::error('Daily bonus error for user ' . $this->id . ': ' . $e->getMessage(), [

                'file' => $e->getFile(),

                'line' => $e->getLine(),

                'user_id' => $this->id,

                'trace' => $e->getTraceAsString()

            ]);

            return false;

        }

    }



    // İlişkiler - Duplicate methods removed, keeping existing ones

    public function watchTasks()

    {

        return $this->hasMany(WatchTask::class, 'created_by');

    }



    public function watchCompletions()

    {

        return $this->hasMany(WatchCompletion::class, 'user_id');

    }

    // Orders relationship
    public function orders()
    {
        return $this->hasMany(Order::class, 'user_id');
    }

     public function getCanBeCancelledAttribute()
     {
      return $this->canBeCancelled();
   }

}

