<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class WatchTask extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'description',
        'youtube_url',
        'youtube_video_id',
        'watch_duration',
        'reward_amount',
        'max_completions',
        'current_completions',
        'is_active',
        'created_by'
    ];

    protected $casts = [
        'reward_amount' => 'decimal:2',
        'is_active' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    protected $appends = ['can_watch', 'is_completed'];


    // İlişkiler

      public function getIsCompletedAttribute()
    {
        return $this->isCompletedByUser(auth()->id());
    }

    public function getCanWatchAttribute()
    {
        return $this->canBeCompleted() && ! $this->isCompletedByUser(auth()->id());
    }
    
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function completions()
    {
        return $this->hasMany(WatchCompletion::class, 'watch_task_id');
    }

    public function userCompletion($userId)
    {
        return $this->hasOne(WatchCompletion::class)->where('user_id', $userId);
    }

    // YouTube video ID'sini URL'den çıkar
    public static function extractVideoId($url)
    {
        $patterns = [
            '/(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/)([a-zA-Z0-9_-]{11})/',
            '/youtube\.com\/shorts\/([a-zA-Z0-9_-]{11})/'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $url, $matches)) {
                return $matches[1];
            }
        }
        
        return null;
    }

    // YouTube embed URL'si oluştur
    public function getEmbedUrlAttribute()
    {
        return "https://www.youtube.com/embed/{$this->youtube_video_id}?enablejsapi=1&origin=" . url('/');
    }

    // Görev tamamlanabilir mi?
    public function canBeCompleted()
    {
        if (!$this->is_active) {
            return false;
        }
        
        if ($this->max_completions > 0 && $this->current_completions >= $this->max_completions) {
            return false;
        }
        
        return true;
    }

    // Kullanıcı bu görevi tamamlamış mı?
    public function isCompletedByUser($userId)
    {
        return $this->completions()->where('user_id', $userId)->where('is_completed', true)->exists();
    }

    // Süre formatı (dakika:saniye)
    public function getFormattedDurationAttribute()
    {
        $minutes = floor($this->watch_duration / 60);
        $seconds = $this->watch_duration % 60;
        return sprintf('%d:%02d', $minutes, $seconds);
    }
}