<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Schema;

class WebsiteVisit extends Model
{
    protected $table = 'website_visits';
    
    protected $fillable = [
        'task_id',
        'user_id',
        'session_id',
        'start_time',
        'end_time',
        'completed_at',
        'duration', // seconds
        'ip_address',
        'user_agent',
        'referrer',
        'status', // 'started', 'in_progress', 'completed', 'failed'
        'reward_given',
        'verification_token'
    ];

    protected $casts = [
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'completed_at' => 'datetime',
        'reward_given' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relationships
    public function task()
    {
        return $this->belongsTo(WebsiteVisitTask::class, 'task_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function proofs()
    {
        return $this->hasMany(WebsiteVisitProof::class, 'visit_id');
    }

    // Scopes
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeToday($query)
    {
        return $query->whereDate('created_at', today());
    }

    // Helper methods
    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    public function getDurationInMinutes()
    {
        return round($this->duration / 60, 2);
    }

    public function markAsCompleted($frontendDuration = null)
    {
        $duration = 0;
        
        // Frontend'den gelen duration'ı öncelikle kullan
        if ($frontendDuration && $frontendDuration > 0) {
            $duration = $frontendDuration;
        } elseif ($this->start_time) {
            // Frontend'den gelmemişse sunucu tarafında hesapla
            $startTime = \Carbon\Carbon::parse($this->start_time);
            $endTime = now();
            $duration = $startTime->diffInSeconds($endTime);
            $duration = max(0, $duration); // Ensure positive
        }
        
        // Debug log
        \Log::info('MARK AS COMPLETED - Duration calculation:', [
            'visit_id' => $this->id,
            'start_time' => $this->start_time,
            'end_time' => now(),
            'frontend_duration' => $frontendDuration,
            'calculated_duration' => $duration,
            'used_frontend' => $frontendDuration ? 'yes' : 'no'
        ]);
        
        // Check if completed_at column exists before using it
        $updateData = [
            'status' => 'completed',
            'end_time' => now(),
            'duration' => $duration
        ];
        
        // Only add completed_at if the column exists
        if (Schema::hasColumn('website_visits', 'completed_at')) {
            $updateData['completed_at'] = now();
        }
        
        $updated = $this->update($updateData);
        
        // Debug log to confirm update
        \Log::info('Visit status update attempt:', [
            'visit_id' => $this->id,
            'update_result' => $updated,
            'new_status' => 'completed',
            'duration' => $duration,
            'current_status_after_update' => $this->fresh()->status
        ]);
    }
}