<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class WebsiteVisitProof extends Model
{
    protected $table = 'website_visit_proofs';
    
    protected $fillable = [
        'visit_id',
        'user_id',
        'description',
        'status', // 0=pending, 1=approved, 2=denied
        'reason',
        'reviewed_by',
        'reviewed_at'
    ];

    protected $casts = [
        'reviewed_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relationships
    public function visit()
    {
        return $this->belongsTo(WebsiteVisit::class, 'visit_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function reviewer()
    {
        return $this->belongsTo(User::class, 'reviewed_by');
    }

    public function files()
    {
        return $this->hasMany(WebsiteVisitProofFile::class, 'proof_id');
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 0);
    }

    public function scopeApproved($query)
    {
        return $query->where('status', 1);
    }

    public function scopeDenied($query)
    {
        return $query->where('status', 2);
    }

    // Helper methods
    public function isPending()
    {
        return $this->status === 0;
    }

    public function isApproved()
    {
        return $this->status === 1;
    }

    public function isDenied()
    {
        return $this->status === 2;
    }

    public function getStatusText()
    {
        return match($this->status) {
            0 => __('Pending'),
            1 => __('Approved'),
            2 => __('Denied'),
            default => __('Unknown')
        };
    }
}