<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class WebsiteVisitTask extends Model
{
    protected $table = 'website_visit_tasks';
    
    protected $fillable = [
        'title',
        'description', 
        'website_url',
        'visit_method', // 'direct' or 'google_search'
        'search_keywords',
        'required_time', // seconds
        'reward_amount',
        'max_completions', // total task limit
        'user_daily_limit', // how many times per day per user
        'user_total_limit', // how many times total per user
        'target_country_id',
        'target_city_id',
        'target_gender', // 0=all, 1=male, 2=female
        'status', // 0=inactive, 1=active, 2=pending
        'created_by',
        'expires_at'
    ];

    protected $casts = [
        'expires_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relationships
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function country()
    {
        return $this->belongsTo(Country::class, 'target_country_id', 'id');
    }

    public function city()
    {
        return $this->belongsTo(City::class, 'target_city_id');
    }

    public function visits()
    {
        return $this->hasMany(WebsiteVisit::class, 'task_id');
    }

    public function completedVisits()
    {
        return $this->hasMany(WebsiteVisit::class, 'task_id')->where('status', 'completed');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeForUser($query, $user)
    {
        return $query->where('created_by', '!=', $user->id) // Görev ekleyen kullanıcı kendi görevini yapamaz
            ->where(function($q) use ($user) {
                // Country targeting
                $q->where('target_country_id', 0)
                  ->orWhereNull('target_country_id');
                if ($user->country_id) {
                    $q->orWhere('target_country_id', $user->country_id);
                }
            })
            ->where(function($q) use ($user) {
                // City targeting  
                $q->where('target_city_id', 0)
                  ->orWhereNull('target_city_id');
                if ($user->city_id) {
                    $q->orWhere('target_city_id', $user->city_id);
                }
            })
            ->where(function($q) use ($user) {
                // Gender targeting
                $q->where('target_gender', 0);
                if ($user->gender) {
                    $q->orWhere('target_gender', $user->gender);
                }
            });
    }

    // Helper methods
    public function canUserComplete($user)
    {
        // Görev ekleyen kullanıcı kendi görevini yapamaz
        if ($this->created_by === $user->id) {
            return false;
        }

        // Check if user has reached daily limit
        $todayVisits = $this->visits()
            ->where('user_id', $user->id)
            ->whereDate('created_at', today())
            ->where('status', 'completed')
            ->count();

        if ($todayVisits >= $this->user_daily_limit) {
            return false;
        }

        // Check if user has reached total limit
        $totalVisits = $this->visits()
            ->where('user_id', $user->id)
            ->where('status', 'completed')
            ->count();

        if ($totalVisits >= $this->user_total_limit) {
            return false;
        }

        // Check if task has reached max completions
        $totalCompletions = $this->completedVisits()->count();
        if ($totalCompletions >= $this->max_completions) {
            return false;
        }

        return true;
    }

    public function getRemainingCompletions()
    {
        return $this->max_completions - $this->completedVisits()->count();
    }

    public function getCompletionPercentage()
    {
        $completed = $this->completedVisits()->count();
        return $this->max_completions > 0 ? ($completed / $this->max_completions) * 100 : 0;
    }
}