<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class WithdrawalRequest extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'amount',
        'iban',
        'account_holder_name',
        'status',
        'admin_note',
        'processed_by',
        'processed_at',
        'transaction_reference',
        'balance_before',
        'balance_after',
        'ip_address',
        'user_agent'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'balance_before' => 'decimal:2',
        'balance_after' => 'decimal:2',
        'processed_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // İlişkiler
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function processedBy()
    {
        return $this->belongsTo(User::class, 'processed_by');
    }

    // Status badge renkleri
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'pending' => 'warning',
            'approved' => 'info',
            'rejected' => 'danger',
            'completed' => 'success'
        ];
        
        return $badges[$this->status] ?? 'secondary';
    }

    // Status metinleri
    public function getStatusTextAttribute()
    {
        $texts = [
            'pending' => __('Pending'),
            'approved' => __('Approved'),
            'rejected' => __('Rejected'),
            'completed' => __('Completed')
        ];
        
        return $texts[$this->status] ?? $this->status;
    }

    // IBAN maskeleme
    public function getMaskedIbanAttribute()
    {
        if (strlen($this->iban) > 8) {
            return substr($this->iban, 0, 4) . str_repeat('*', strlen($this->iban) - 8) . substr($this->iban, -4);
        }
        return $this->iban;
    }

    // Scope'lar
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeRejected($query)
    {
        return $query->where('status', 'rejected');
    }
}