<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Gate;
use App\Services\BackupService;
use App\Models\Backup;
use App\Policies\BackupPolicy;

class BackupServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        // Register backup disk configuration first
        $this->registerBackupDisk();

        // Register the backup service
        $this->app->singleton('backup.service', function ($app) {
            return new BackupService();
        });

        // Register the facade alias
        $this->app->alias('backup.service', BackupService::class);

        // Register helper functions
        $this->registerHelpers();
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Register policies
        Gate::policy(Backup::class, BackupPolicy::class);

        // Register middleware
        $this->app['router']->aliasMiddleware('backup', \App\Http\Middleware\BackupMiddleware::class);

        // Publish configuration
        if ($this->app->runningInConsole()) {
            $this->publishes([
                __DIR__.'/../../config/backup.php' => config_path('backup.php'),
            ], 'backup-config');
        }

        // Register commands if running in console
        if ($this->app->runningInConsole()) {
            $this->commands([
                \App\Console\Commands\BackupCreateCommand::class,
                \App\Console\Commands\BackupCleanupCommand::class,
                \App\Console\Commands\BackupListCommand::class,
            ]);
        }
    }

    /**
     * Register backup disk configuration
     */
    protected function registerBackupDisk(): void
    {
        // Ensure backup directory exists
        $backupPath = storage_path('app/backups');
        if (!file_exists($backupPath)) {
            mkdir($backupPath, 0755, true);
        }

        // Register backup disk configuration
        config([
            'filesystems.disks.backups' => [
                'driver' => 'local',
                'root' => $backupPath,
                'url' => env('APP_URL').'/storage/backups',
                'visibility' => 'private',
                'permissions' => [
                    'file' => [
                        'public' => 0644,
                        'private' => 0600,
                    ],
                    'dir' => [
                        'public' => 0755,
                        'private' => 0700,
                    ],
                ],
            ]
        ]);
    }

    /**
     * Register helper functions
     */
    protected function registerHelpers(): void
    {
        if (!function_exists('formatBytes')) {
            function formatBytes($size, $precision = 2) {
                return \App\Helpers\BackupHelper::formatBytes($size, $precision);
            }
        }

        if (!function_exists('backup')) {
            function backup() {
                return app('backup.service');
            }
        }
    }
}